/*
 * Decompiled with CFR 0.152.
 */
package org.keycloak.services.util;

import jakarta.ws.rs.core.UriBuilder;
import jakarta.ws.rs.core.UriInfo;
import java.net.URI;
import java.net.URISyntaxException;
import java.util.Set;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import org.apache.commons.codec.binary.Hex;
import org.keycloak.TokenVerifier;
import org.keycloak.common.VerificationException;
import org.keycloak.common.util.Time;
import org.keycloak.crypto.KeyWrapper;
import org.keycloak.crypto.SignatureProvider;
import org.keycloak.crypto.SignatureVerifierContext;
import org.keycloak.exceptions.TokenVerificationException;
import org.keycloak.http.HttpRequest;
import org.keycloak.jose.jwk.JWK;
import org.keycloak.jose.jws.JWSHeader;
import org.keycloak.jose.jws.crypto.HashUtils;
import org.keycloak.models.KeycloakSession;
import org.keycloak.models.SingleUseObjectProvider;
import org.keycloak.representations.AccessToken;
import org.keycloak.representations.JsonWebToken;
import org.keycloak.representations.dpop.DPoP;
import org.keycloak.util.JWKSUtils;
import org.keycloak.utils.StringUtil;

public class DPoPUtil {
    public static final int DEFAULT_PROOF_LIFETIME = 10;
    public static final int DEFAULT_ALLOWED_CLOCK_SKEW = 2;
    public static final String DPOP_TOKEN_TYPE = "DPoP";
    public static final String DPOP_SCHEME = "DPoP";
    public static final String DPOP_SESSION_ATTRIBUTE = "dpop";
    public static final String DPOP_PARAM = "dpop";
    public static final String DPOP_THUMBPRINT_NOTE = "dpop.thumbprint";
    public static final String DPOP_HTTP_HEADER = "DPoP";
    private static final String DPOP_JWT_HEADER_TYPE = "dpop+jwt";
    private static final String DPOP_ATH_ALG = "RS256";
    public static final Set<String> DPOP_SUPPORTED_ALGS = Stream.of("ES256", "ES384", "ES512", "PS256", "PS384", "PS512", "RS256", "RS384", "RS512").collect(Collectors.toSet());

    private static URI normalize(URI uri) {
        return UriBuilder.fromUri((URI)uri).replaceQuery("").build(new Object[0]);
    }

    private static DPoP validateDPoP(KeycloakSession session, URI uri, String method, String token, String accessToken, int lifetime, int clockSkew) throws VerificationException {
        JWSHeader header;
        if (token == null || token.trim().equals("")) {
            throw new VerificationException("DPoP proof is missing");
        }
        TokenVerifier verifier = TokenVerifier.create((String)token, DPoP.class);
        try {
            header = verifier.getHeader();
        }
        catch (VerificationException ex) {
            throw new VerificationException("DPoP header verification failure");
        }
        if (!DPOP_JWT_HEADER_TYPE.equals(header.getType())) {
            throw new VerificationException("Invalid or missing type in DPoP header: " + header.getType());
        }
        String algorithm = header.getAlgorithm().name();
        if (!DPOP_SUPPORTED_ALGS.contains(algorithm)) {
            throw new VerificationException("Unsupported DPoP algorithm: " + String.valueOf(header.getAlgorithm()));
        }
        JWK jwk = header.getKey();
        if (jwk == null) {
            throw new VerificationException("No JWK in DPoP header");
        }
        KeyWrapper key = JWKSUtils.getKeyWrapper((JWK)jwk);
        if (key.getPublicKey() == null) {
            throw new VerificationException("No public key in DPoP header");
        }
        if (key.getPrivateKey() != null) {
            throw new VerificationException("Private key is present in DPoP header");
        }
        key.setAlgorithm(header.getAlgorithm().name());
        SignatureVerifierContext signatureVerifier = ((SignatureProvider)session.getProvider(SignatureProvider.class, algorithm)).verifier(key);
        verifier.verifierContext(signatureVerifier);
        verifier.withChecks(new TokenVerifier.Predicate[]{DPoPClaimsCheck.INSTANCE, new DPoPHTTPCheck(uri, method), new DPoPIsActiveCheck(session, lifetime, clockSkew), new DPoPReplayCheck(session, lifetime)});
        if (accessToken != null) {
            verifier.withChecks(new TokenVerifier.Predicate[]{new DPoPAccessTokenHashCheck(accessToken)});
        }
        try {
            DPoP dPoP = (DPoP)verifier.verify().getToken();
            dPoP.setThumbprint(JWKSUtils.computeThumbprint((JWK)jwk));
            return dPoP;
        }
        catch (DPoPVerificationException ex) {
            throw ex;
        }
        catch (VerificationException ex) {
            throw new VerificationException("DPoP verification failure", (Throwable)ex);
        }
    }

    public static void validateBinding(AccessToken token, DPoP dPoP) throws VerificationException {
        try {
            TokenVerifier.createWithoutSignature((JsonWebToken)token).withChecks(new TokenVerifier.Predicate[]{new DPoPBindingCheck(dPoP)}).verify();
        }
        catch (TokenVerificationException ex) {
            throw ex;
        }
        catch (VerificationException ex) {
            throw new VerificationException("Token verification failure", (Throwable)ex);
        }
    }

    public static void bindToken(AccessToken token, String thumbprint) {
        AccessToken.Confirmation confirmation = token.getConfirmation();
        if (confirmation == null) {
            confirmation = new AccessToken.Confirmation();
            token.setConfirmation(confirmation);
        }
        confirmation.setKeyThumbprint(thumbprint);
    }

    public static void bindToken(AccessToken token, DPoP dPoP) {
        DPoPUtil.bindToken(token, dPoP.getThumbprint());
    }

    private static class DPoPClaimsCheck
    implements TokenVerifier.Predicate<DPoP> {
        static final TokenVerifier.Predicate<DPoP> INSTANCE = new DPoPClaimsCheck();

        private DPoPClaimsCheck() {
        }

        public boolean test(DPoP t) throws DPoPVerificationException {
            Long iat = t.getIat();
            String jti = t.getId();
            String htu = t.getHttpUri();
            String htm = t.getHttpMethod();
            if (iat != null && StringUtil.isNotBlank((String)jti) && StringUtil.isNotBlank((String)htm) && StringUtil.isNotBlank((String)htu)) {
                return true;
            }
            throw new DPoPVerificationException(t, "DPoP mandatory claims are missing");
        }
    }

    private static class DPoPHTTPCheck
    implements TokenVerifier.Predicate<DPoP> {
        private final URI uri;
        private final String method;

        DPoPHTTPCheck(URI uri, String method) {
            this.uri = uri;
            this.method = method;
        }

        public boolean test(DPoP t) throws DPoPVerificationException {
            try {
                if (!DPoPUtil.normalize(new URI(t.getHttpUri())).equals(DPoPUtil.normalize(this.uri))) {
                    throw new DPoPVerificationException(t, "DPoP HTTP URL mismatch");
                }
                if (!this.method.equals(t.getHttpMethod())) {
                    throw new DPoPVerificationException(t, "DPoP HTTP method mismatch");
                }
            }
            catch (URISyntaxException ex) {
                throw new DPoPVerificationException(t, "Malformed HTTP URL in DPoP proof");
            }
            return true;
        }
    }

    private static class DPoPIsActiveCheck
    implements TokenVerifier.Predicate<DPoP> {
        private final int lifetime;
        private final int clockSkew;

        public DPoPIsActiveCheck(KeycloakSession session, int lifetime, int clockSkew) {
            this.lifetime = lifetime;
            this.clockSkew = clockSkew;
        }

        public boolean test(DPoP t) throws DPoPVerificationException {
            long time = Time.currentTime();
            Long iat = t.getIat();
            if (iat > time + (long)this.clockSkew || iat <= time - (long)this.lifetime) {
                throw new DPoPVerificationException(t, "DPoP proof is not active");
            }
            return true;
        }
    }

    private static class DPoPReplayCheck
    implements TokenVerifier.Predicate<DPoP> {
        private final KeycloakSession session;
        private final int lifetime;

        public DPoPReplayCheck(KeycloakSession session, int lifetime) {
            this.session = session;
            this.lifetime = lifetime;
        }

        public boolean test(DPoP t) throws DPoPVerificationException {
            byte[] hash;
            String hashString;
            SingleUseObjectProvider singleUseCache = this.session.singleUseObjects();
            if (!singleUseCache.putIfAbsent(hashString = Hex.encodeHexString((byte[])(hash = HashUtils.hash((String)"SHA1", (byte[])(t.getId() + "\n" + t.getHttpUri()).getBytes()))), (long)((int)(t.getIat() + (long)this.lifetime - (long)Time.currentTime())))) {
                throw new DPoPVerificationException(t, "DPoP proof has already been used");
            }
            return true;
        }
    }

    private static class DPoPAccessTokenHashCheck
    implements TokenVerifier.Predicate<DPoP> {
        private String hash;

        public DPoPAccessTokenHashCheck(String tokenString) {
            this.hash = HashUtils.accessTokenHash((String)DPoPUtil.DPOP_ATH_ALG, (String)tokenString, (boolean)true);
        }

        public boolean test(DPoP t) throws DPoPVerificationException {
            if (t.getAccessTokenHash() == null) {
                throw new DPoPVerificationException(t, "No access token hash in DPoP proof");
            }
            if (!t.getAccessTokenHash().equals(this.hash)) {
                throw new DPoPVerificationException(t, "DPoP proof access token hash mismatch");
            }
            return true;
        }
    }

    public static class DPoPVerificationException
    extends TokenVerificationException {
        public DPoPVerificationException(DPoP token, String message) {
            super((JsonWebToken)token, message);
        }
    }

    private static class DPoPBindingCheck
    implements TokenVerifier.Predicate<AccessToken> {
        private final DPoP proof;

        public DPoPBindingCheck(DPoP proof) {
            this.proof = proof;
        }

        public boolean test(AccessToken t) throws VerificationException {
            String thumbprint = this.proof.getThumbprint();
            AccessToken.Confirmation confirmation = t.getConfirmation();
            if (confirmation == null) {
                throw new TokenVerificationException((JsonWebToken)t, "No DPoP confirmation in access token");
            }
            String keyThumbprint = confirmation.getKeyThumbprint();
            if (keyThumbprint == null) {
                throw new TokenVerificationException((JsonWebToken)t, "No DPoP key thumbprint in access token");
            }
            if (!keyThumbprint.equals(thumbprint)) {
                throw new TokenVerificationException((JsonWebToken)t, "DPoP confirmation doesn't match DPoP proof");
            }
            return true;
        }
    }

    public static class Validator {
        private URI uri;
        private String method;
        private String dPoP;
        private String accessToken;
        private int clockSkew = 2;
        private int lifetime = 10;
        private final KeycloakSession session;

        public Validator(KeycloakSession session) {
            this.session = session;
        }

        public Validator request(HttpRequest request) {
            this.uri = request.getUri().getAbsolutePath();
            this.method = request.getHttpMethod();
            this.dPoP = request.getHttpHeaders().getHeaderString("DPoP");
            return this;
        }

        public Validator dPoP(String dPoP) {
            this.dPoP = dPoP;
            return this;
        }

        public Validator accessToken(String accessToken) {
            this.accessToken = accessToken;
            return this;
        }

        public Validator uriInfo(UriInfo uriInfo) {
            this.uri = uriInfo.getAbsolutePath();
            return this;
        }

        public Validator uri(String uri) throws URISyntaxException {
            this.uri = new URI(uri);
            return this;
        }

        public Validator method(String method) {
            this.method = method;
            return this;
        }

        public DPoP validate() throws VerificationException {
            return DPoPUtil.validateDPoP(this.session, this.uri, this.method, this.dPoP, this.accessToken, this.lifetime, this.clockSkew);
        }
    }

    public static enum Mode {
        ENABLED,
        OPTIONAL,
        DISABLED;

    }
}

