/*
 * Decompiled with CFR 0.152.
 */
package org.keycloak.utils;

import java.security.GeneralSecurityException;
import java.security.cert.Certificate;
import java.security.cert.X509CRL;
import java.security.cert.X509Certificate;
import java.util.Arrays;
import java.util.Map;
import java.util.Set;
import java.util.stream.Collectors;
import javax.security.auth.x500.X500Principal;
import org.jboss.logging.Logger;
import org.keycloak.models.KeycloakSession;
import org.keycloak.truststore.TruststoreProvider;

public final class CRLUtils {
    private static final Logger log = Logger.getLogger(CRLUtils.class);

    public static void check(X509Certificate[] certs, X509CRL crl, KeycloakSession session) throws GeneralSecurityException {
        if (certs == null || certs.length < 1) {
            throw new GeneralSecurityException("Not possible to verify signature on CRL because no certificate chain was passed.");
        }
        X500Principal crlIssuerPrincipal = crl.getIssuerX500Principal();
        Certificate crlSignatureCertificate = null;
        for (X509Certificate currentCACert : certs) {
            if (!crlIssuerPrincipal.equals(currentCACert.getSubjectX500Principal())) continue;
            crlSignatureCertificate = currentCACert;
            log.tracef("Found certificate used to sign CRL in the CA chain of the certificate. CRL issuer: %s", (Object)crlIssuerPrincipal);
            break;
        }
        if (crlSignatureCertificate == null) {
            log.tracef("Not found CRL issuer '%s' in the CA chain of the certificate. Fallback to lookup CRL issuer in the truststore", (Object)crlIssuerPrincipal);
            crlSignatureCertificate = CRLUtils.findCRLSignatureCertificateInTruststore(session, certs, crlIssuerPrincipal);
        }
        crl.verify(crlSignatureCertificate.getPublicKey());
        if (crl.isRevoked(certs[0])) {
            String message = String.format("Certificate has been revoked, certificate's subject: %s", certs[0].getSubjectDN().getName());
            log.debug((Object)message);
            throw new GeneralSecurityException(message);
        }
    }

    private static X509Certificate findCRLSignatureCertificateInTruststore(KeycloakSession session, X509Certificate[] certs, X500Principal crlIssuerPrincipal) throws GeneralSecurityException {
        TruststoreProvider truststoreProvider = (TruststoreProvider)session.getProvider(TruststoreProvider.class);
        if (truststoreProvider == null || truststoreProvider.getTruststore() == null) {
            throw new GeneralSecurityException("Truststore not available");
        }
        Map rootCerts = truststoreProvider.getRootCertificates();
        Map intermediateCerts = truststoreProvider.getIntermediateCertificates();
        X509Certificate crlSignatureCertificate = (X509Certificate)intermediateCerts.get(crlIssuerPrincipal);
        if (crlSignatureCertificate == null) {
            crlSignatureCertificate = (X509Certificate)rootCerts.get(crlIssuerPrincipal);
        }
        if (crlSignatureCertificate == null) {
            throw new GeneralSecurityException("Not available certificate for CRL issuer '" + String.valueOf(crlIssuerPrincipal) + "' in the truststore, nor in the CA chain");
        }
        log.tracef("Found CRL issuer certificate with subject '%s' in the truststore. Verifying trust anchor", (Object)crlIssuerPrincipal);
        Set certificateCAPrincipals = Arrays.asList(certs).stream().map(X509Certificate::getIssuerX500Principal).collect(Collectors.toSet());
        X509Certificate currentCRLAnchorCertificate = crlSignatureCertificate;
        X500Principal currentCRLAnchorPrincipal = crlIssuerPrincipal;
        for (X500Principal certificateCAPrincipal : certificateCAPrincipals) {
            if (certificateCAPrincipal.equals(currentCRLAnchorPrincipal)) {
                log.tracef("Found trust anchor of the CRL issuer '%s' in the CA chain. Anchor is '%s'", (Object)crlIssuerPrincipal, (Object)currentCRLAnchorPrincipal);
                break;
            }
            currentCRLAnchorPrincipal = currentCRLAnchorCertificate.getIssuerX500Principal();
            if ((currentCRLAnchorCertificate = (X509Certificate)intermediateCerts.get(currentCRLAnchorPrincipal)) == null) {
                currentCRLAnchorCertificate = (X509Certificate)rootCerts.get(currentCRLAnchorPrincipal);
            }
            if (currentCRLAnchorCertificate != null) continue;
            throw new GeneralSecurityException("Certificate for CRL issuer '" + String.valueOf(crlIssuerPrincipal) + "' available in the truststore, but doesn't have trust anchors with the CA chain.");
        }
        return crlSignatureCertificate;
    }
}

