"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.PanelsRouter = PanelsRouter;
var _configSchema = require("@osd/config-schema");
var _custom_panel_adaptor = require("../../adaptors/custom_panels/custom_panel_adaptor");
var _custom_panels = require("../../../common/constants/custom_panels");
/*
 * Copyright OpenSearch Contributors
 * SPDX-License-Identifier: Apache-2.0
 */

function PanelsRouter(router) {
  const customPanelBackend = new _custom_panel_adaptor.CustomPanelsAdaptor();
  // Fetch all the custom panels available
  router.get({
    path: `${_custom_panels.CUSTOM_PANELS_API_PREFIX}/panels`,
    validate: {}
  }, async (context, request, response) => {
    const opensearchNotebooksClient = context.observability_plugin.observabilityClient.asScoped(request);
    try {
      const panelsList = await customPanelBackend.viewPanelList(opensearchNotebooksClient);
      return response.ok({
        body: {
          panels: panelsList
        }
      });
    } catch (error) {
      console.error('Issue in fetching panel list:', error);
      return response.custom({
        statusCode: error.statusCode || 500,
        body: error.message
      });
    }
  });

  // Fetch the required panel by id
  router.get({
    path: `${_custom_panels.CUSTOM_PANELS_API_PREFIX}/panels/{panelId}`,
    validate: {
      params: _configSchema.schema.object({
        panelId: _configSchema.schema.string()
      })
    }
  }, async (context, request, response) => {
    const opensearchNotebooksClient = context.observability_plugin.observabilityClient.asScoped(request);
    try {
      const panelObject = await customPanelBackend.getPanel(opensearchNotebooksClient, request.params.panelId);
      return response.ok({
        body: panelObject
      });
    } catch (error) {
      console.error('Issue in fetching panel:', error);
      return response.custom({
        statusCode: error.statusCode || 500,
        body: error.message
      });
    }
  });

  // Create a new panel
  router.post({
    path: `${_custom_panels.CUSTOM_PANELS_API_PREFIX}/panels`,
    validate: {
      body: _configSchema.schema.object({
        panelName: _configSchema.schema.string(),
        applicationId: _configSchema.schema.maybe(_configSchema.schema.string())
      })
    }
  }, async (context, request, response) => {
    const opensearchNotebooksClient = context.observability_plugin.observabilityClient.asScoped(request);
    try {
      const newPanelId = await customPanelBackend.createNewPanel(opensearchNotebooksClient, request.body.panelName, request.body.applicationId || '');
      return response.ok({
        body: {
          message: 'Panel Created',
          newPanelId
        }
      });
    } catch (error) {
      console.error('Issue in creating new panel', error);
      return response.custom({
        statusCode: error.statusCode || 500,
        body: error.message
      });
    }
  });

  // update an existing panel
  router.post({
    path: `${_custom_panels.CUSTOM_PANELS_API_PREFIX}/panels/update`,
    validate: {
      body: _configSchema.schema.object({
        panelId: _configSchema.schema.string(),
        panel: _configSchema.schema.any()
      })
    }
  }, async (context, request, response) => {
    const opensearchNotebooksClient = context.observability_plugin.observabilityClient.asScoped(request);
    try {
      const responseBody = await customPanelBackend.updatePanel(opensearchNotebooksClient, request.body.panelId, request.body.panel);
      return response.ok({
        body: {
          message: 'Panel Updated'
        }
      });
    } catch (error) {
      console.error('Issue in updating panel', error);
      return response.custom({
        statusCode: error.statusCode || 500,
        body: error.message
      });
    }
  });

  // rename an existing panel
  router.post({
    path: `${_custom_panels.CUSTOM_PANELS_API_PREFIX}/panels/rename`,
    validate: {
      body: _configSchema.schema.object({
        panelId: _configSchema.schema.string(),
        panelName: _configSchema.schema.string()
      })
    }
  }, async (context, request, response) => {
    const opensearchNotebooksClient = context.observability_plugin.observabilityClient.asScoped(request);
    try {
      const responseBody = await customPanelBackend.renamePanel(opensearchNotebooksClient, request.body.panelId, request.body.panelName);
      return response.ok({
        body: {
          message: 'Panel Renamed'
        }
      });
    } catch (error) {
      console.error('Issue in renaming panel', error);
      return response.custom({
        statusCode: error.statusCode || 500,
        body: error.message
      });
    }
  });

  // clones an existing panel
  // returns new panel Id
  router.post({
    path: `${_custom_panels.CUSTOM_PANELS_API_PREFIX}/panels/clone`,
    validate: {
      body: _configSchema.schema.object({
        panelId: _configSchema.schema.string(),
        panelName: _configSchema.schema.string()
      })
    }
  }, async (context, request, response) => {
    const opensearchNotebooksClient = context.observability_plugin.observabilityClient.asScoped(request);
    try {
      const cloneResponse = await customPanelBackend.clonePanel(opensearchNotebooksClient, request.body.panelId, request.body.panelName);
      return response.ok({
        body: {
          message: 'Panel Cloned',
          clonePanelId: cloneResponse.clonePanelId,
          dateCreated: cloneResponse.dateCreated,
          dateModified: cloneResponse.dateModified
        }
      });
    } catch (error) {
      console.error('Issue in cloning panel', error);
      return response.custom({
        statusCode: error.statusCode || 500,
        body: error.message
      });
    }
  });

  // delete an existing panel
  router.delete({
    path: `${_custom_panels.CUSTOM_PANELS_API_PREFIX}/panels/{panelId}`,
    validate: {
      params: _configSchema.schema.object({
        panelId: _configSchema.schema.string()
      })
    }
  }, async (context, request, response) => {
    const opensearchNotebooksClient = context.observability_plugin.observabilityClient.asScoped(request);
    try {
      const deleteResponse = await customPanelBackend.deletePanel(opensearchNotebooksClient, request.params.panelId);
      return response.ok({
        body: {
          message: 'Panel Deleted'
        }
      });
    } catch (error) {
      console.error('Issue in deleting panel', error);
      return response.custom({
        statusCode: error.statusCode || 500,
        body: error.message
      });
    }
  });

  // delete an existing panel(s)
  router.delete({
    path: `${_custom_panels.CUSTOM_PANELS_API_PREFIX}/panelList/{panelIdList}`,
    validate: {
      params: _configSchema.schema.object({
        panelIdList: _configSchema.schema.string()
      })
    }
  }, async (context, request, response) => {
    const opensearchNotebooksClient = context.observability_plugin.observabilityClient.asScoped(request);
    try {
      const deleteResponse = await customPanelBackend.deletePanelList(opensearchNotebooksClient, request.params.panelIdList);
      return response.ok({
        body: {
          message: 'Panel Deleted'
        }
      });
    } catch (error) {
      console.error('Issue in deleting panel', error);
      return response.custom({
        statusCode: error.statusCode || 500,
        body: error.message
      });
    }
  });

  // replaces the ppl query filter in panel
  router.post({
    path: `${_custom_panels.CUSTOM_PANELS_API_PREFIX}/panels/filter`,
    validate: {
      body: _configSchema.schema.object({
        panelId: _configSchema.schema.string(),
        query: _configSchema.schema.string(),
        language: _configSchema.schema.string(),
        to: _configSchema.schema.string(),
        from: _configSchema.schema.string()
      })
    }
  }, async (context, request, response) => {
    const opensearchNotebooksClient = context.observability_plugin.observabilityClient.asScoped(request);
    try {
      const panelFilterResponse = await customPanelBackend.addPanelFilter(opensearchNotebooksClient, request.body.panelId, request.body.query, request.body.language, request.body.to, request.body.from);
      return response.ok({
        body: {
          message: 'Panel PPL Filter Changed'
        }
      });
    } catch (error) {
      console.error('Issue in adding query filter', error);
      return response.custom({
        statusCode: error.statusCode || 500,
        body: error.message
      });
    }
  });

  // Add Sample Panels
  router.post({
    path: `${_custom_panels.CUSTOM_PANELS_API_PREFIX}/panels/addSamplePanels`,
    validate: {
      body: _configSchema.schema.object({
        savedVisualizationIds: _configSchema.schema.arrayOf(_configSchema.schema.string())
      })
    }
  }, async (context, request, response) => {
    const opensearchNotebooksClient = context.observability_plugin.observabilityClient.asScoped(request);
    try {
      const panelsData = await customPanelBackend.addSamplePanels(opensearchNotebooksClient, request.body.savedVisualizationIds);
      return response.ok({
        body: {
          demoPanelsData: panelsData
        }
      });
    } catch (error) {
      console.error('Issue in fetching panel list:', error);
      return response.custom({
        statusCode: error.statusCode || 500,
        body: error.message
      });
    }
  });
}
//# sourceMappingURL=data:application/json;charset=utf-8;base64,eyJ2ZXJzaW9uIjozLCJuYW1lcyI6WyJfY29uZmlnU2NoZW1hIiwicmVxdWlyZSIsIl9jdXN0b21fcGFuZWxfYWRhcHRvciIsIl9jdXN0b21fcGFuZWxzIiwiUGFuZWxzUm91dGVyIiwicm91dGVyIiwiY3VzdG9tUGFuZWxCYWNrZW5kIiwiQ3VzdG9tUGFuZWxzQWRhcHRvciIsImdldCIsInBhdGgiLCJBUElfUFJFRklYIiwidmFsaWRhdGUiLCJjb250ZXh0IiwicmVxdWVzdCIsInJlc3BvbnNlIiwib3BlbnNlYXJjaE5vdGVib29rc0NsaWVudCIsIm9ic2VydmFiaWxpdHlfcGx1Z2luIiwib2JzZXJ2YWJpbGl0eUNsaWVudCIsImFzU2NvcGVkIiwicGFuZWxzTGlzdCIsInZpZXdQYW5lbExpc3QiLCJvayIsImJvZHkiLCJwYW5lbHMiLCJlcnJvciIsImNvbnNvbGUiLCJjdXN0b20iLCJzdGF0dXNDb2RlIiwibWVzc2FnZSIsInBhcmFtcyIsInNjaGVtYSIsIm9iamVjdCIsInBhbmVsSWQiLCJzdHJpbmciLCJwYW5lbE9iamVjdCIsImdldFBhbmVsIiwicG9zdCIsInBhbmVsTmFtZSIsImFwcGxpY2F0aW9uSWQiLCJtYXliZSIsIm5ld1BhbmVsSWQiLCJjcmVhdGVOZXdQYW5lbCIsInBhbmVsIiwiYW55IiwicmVzcG9uc2VCb2R5IiwidXBkYXRlUGFuZWwiLCJyZW5hbWVQYW5lbCIsImNsb25lUmVzcG9uc2UiLCJjbG9uZVBhbmVsIiwiY2xvbmVQYW5lbElkIiwiZGF0ZUNyZWF0ZWQiLCJkYXRlTW9kaWZpZWQiLCJkZWxldGUiLCJkZWxldGVSZXNwb25zZSIsImRlbGV0ZVBhbmVsIiwicGFuZWxJZExpc3QiLCJkZWxldGVQYW5lbExpc3QiLCJxdWVyeSIsImxhbmd1YWdlIiwidG8iLCJmcm9tIiwicGFuZWxGaWx0ZXJSZXNwb25zZSIsImFkZFBhbmVsRmlsdGVyIiwic2F2ZWRWaXN1YWxpemF0aW9uSWRzIiwiYXJyYXlPZiIsInBhbmVsc0RhdGEiLCJhZGRTYW1wbGVQYW5lbHMiLCJkZW1vUGFuZWxzRGF0YSJdLCJzb3VyY2VzIjpbInBhbmVsc19yb3V0ZXIudHMiXSwic291cmNlc0NvbnRlbnQiOlsiLypcbiAqIENvcHlyaWdodCBPcGVuU2VhcmNoIENvbnRyaWJ1dG9yc1xuICogU1BEWC1MaWNlbnNlLUlkZW50aWZpZXI6IEFwYWNoZS0yLjBcbiAqL1xuXG5pbXBvcnQgeyBzY2hlbWEgfSBmcm9tICdAb3NkL2NvbmZpZy1zY2hlbWEnO1xuaW1wb3J0IHsgQ3VzdG9tUGFuZWxzQWRhcHRvciB9IGZyb20gJy4uLy4uL2FkYXB0b3JzL2N1c3RvbV9wYW5lbHMvY3VzdG9tX3BhbmVsX2FkYXB0b3InO1xuaW1wb3J0IHtcbiAgSVJvdXRlcixcbiAgSU9wZW5TZWFyY2hEYXNoYm9hcmRzUmVzcG9uc2UsXG4gIFJlc3BvbnNlRXJyb3IsXG4gIElMZWdhY3lTY29wZWRDbHVzdGVyQ2xpZW50LFxufSBmcm9tICcuLi8uLi8uLi8uLi8uLi9zcmMvY29yZS9zZXJ2ZXInO1xuaW1wb3J0IHsgQ1VTVE9NX1BBTkVMU19BUElfUFJFRklYIGFzIEFQSV9QUkVGSVggfSBmcm9tICcuLi8uLi8uLi9jb21tb24vY29uc3RhbnRzL2N1c3RvbV9wYW5lbHMnO1xuXG5leHBvcnQgZnVuY3Rpb24gUGFuZWxzUm91dGVyKHJvdXRlcjogSVJvdXRlcikge1xuICBjb25zdCBjdXN0b21QYW5lbEJhY2tlbmQgPSBuZXcgQ3VzdG9tUGFuZWxzQWRhcHRvcigpO1xuICAvLyBGZXRjaCBhbGwgdGhlIGN1c3RvbSBwYW5lbHMgYXZhaWxhYmxlXG4gIHJvdXRlci5nZXQoXG4gICAge1xuICAgICAgcGF0aDogYCR7QVBJX1BSRUZJWH0vcGFuZWxzYCxcbiAgICAgIHZhbGlkYXRlOiB7fSxcbiAgICB9LFxuICAgIGFzeW5jIChcbiAgICAgIGNvbnRleHQsXG4gICAgICByZXF1ZXN0LFxuICAgICAgcmVzcG9uc2VcbiAgICApOiBQcm9taXNlPElPcGVuU2VhcmNoRGFzaGJvYXJkc1Jlc3BvbnNlPGFueSB8IFJlc3BvbnNlRXJyb3I+PiA9PiB7XG4gICAgICBjb25zdCBvcGVuc2VhcmNoTm90ZWJvb2tzQ2xpZW50OiBJTGVnYWN5U2NvcGVkQ2x1c3RlckNsaWVudCA9IGNvbnRleHQub2JzZXJ2YWJpbGl0eV9wbHVnaW4ub2JzZXJ2YWJpbGl0eUNsaWVudC5hc1Njb3BlZChcbiAgICAgICAgcmVxdWVzdFxuICAgICAgKTtcblxuICAgICAgdHJ5IHtcbiAgICAgICAgY29uc3QgcGFuZWxzTGlzdCA9IGF3YWl0IGN1c3RvbVBhbmVsQmFja2VuZC52aWV3UGFuZWxMaXN0KG9wZW5zZWFyY2hOb3RlYm9va3NDbGllbnQpO1xuICAgICAgICByZXR1cm4gcmVzcG9uc2Uub2soe1xuICAgICAgICAgIGJvZHk6IHtcbiAgICAgICAgICAgIHBhbmVsczogcGFuZWxzTGlzdCxcbiAgICAgICAgICB9LFxuICAgICAgICB9KTtcbiAgICAgIH0gY2F0Y2ggKGVycm9yOiBhbnkpIHtcbiAgICAgICAgY29uc29sZS5lcnJvcignSXNzdWUgaW4gZmV0Y2hpbmcgcGFuZWwgbGlzdDonLCBlcnJvcik7XG4gICAgICAgIHJldHVybiByZXNwb25zZS5jdXN0b20oe1xuICAgICAgICAgIHN0YXR1c0NvZGU6IGVycm9yLnN0YXR1c0NvZGUgfHwgNTAwLFxuICAgICAgICAgIGJvZHk6IGVycm9yLm1lc3NhZ2UsXG4gICAgICAgIH0pO1xuICAgICAgfVxuICAgIH1cbiAgKTtcblxuICAvLyBGZXRjaCB0aGUgcmVxdWlyZWQgcGFuZWwgYnkgaWRcbiAgcm91dGVyLmdldChcbiAgICB7XG4gICAgICBwYXRoOiBgJHtBUElfUFJFRklYfS9wYW5lbHMve3BhbmVsSWR9YCxcbiAgICAgIHZhbGlkYXRlOiB7XG4gICAgICAgIHBhcmFtczogc2NoZW1hLm9iamVjdCh7XG4gICAgICAgICAgcGFuZWxJZDogc2NoZW1hLnN0cmluZygpLFxuICAgICAgICB9KSxcbiAgICAgIH0sXG4gICAgfSxcbiAgICBhc3luYyAoXG4gICAgICBjb250ZXh0LFxuICAgICAgcmVxdWVzdCxcbiAgICAgIHJlc3BvbnNlXG4gICAgKTogUHJvbWlzZTxJT3BlblNlYXJjaERhc2hib2FyZHNSZXNwb25zZTxhbnkgfCBSZXNwb25zZUVycm9yPj4gPT4ge1xuICAgICAgY29uc3Qgb3BlbnNlYXJjaE5vdGVib29rc0NsaWVudDogSUxlZ2FjeVNjb3BlZENsdXN0ZXJDbGllbnQgPSBjb250ZXh0Lm9ic2VydmFiaWxpdHlfcGx1Z2luLm9ic2VydmFiaWxpdHlDbGllbnQuYXNTY29wZWQoXG4gICAgICAgIHJlcXVlc3RcbiAgICAgICk7XG5cbiAgICAgIHRyeSB7XG4gICAgICAgIGNvbnN0IHBhbmVsT2JqZWN0ID0gYXdhaXQgY3VzdG9tUGFuZWxCYWNrZW5kLmdldFBhbmVsKFxuICAgICAgICAgIG9wZW5zZWFyY2hOb3RlYm9va3NDbGllbnQsXG4gICAgICAgICAgcmVxdWVzdC5wYXJhbXMucGFuZWxJZFxuICAgICAgICApO1xuICAgICAgICByZXR1cm4gcmVzcG9uc2Uub2soe1xuICAgICAgICAgIGJvZHk6IHBhbmVsT2JqZWN0LFxuICAgICAgICB9KTtcbiAgICAgIH0gY2F0Y2ggKGVycm9yOiBhbnkpIHtcbiAgICAgICAgY29uc29sZS5lcnJvcignSXNzdWUgaW4gZmV0Y2hpbmcgcGFuZWw6JywgZXJyb3IpO1xuICAgICAgICByZXR1cm4gcmVzcG9uc2UuY3VzdG9tKHtcbiAgICAgICAgICBzdGF0dXNDb2RlOiBlcnJvci5zdGF0dXNDb2RlIHx8IDUwMCxcbiAgICAgICAgICBib2R5OiBlcnJvci5tZXNzYWdlLFxuICAgICAgICB9KTtcbiAgICAgIH1cbiAgICB9XG4gICk7XG5cbiAgLy8gQ3JlYXRlIGEgbmV3IHBhbmVsXG4gIHJvdXRlci5wb3N0KFxuICAgIHtcbiAgICAgIHBhdGg6IGAke0FQSV9QUkVGSVh9L3BhbmVsc2AsXG4gICAgICB2YWxpZGF0ZToge1xuICAgICAgICBib2R5OiBzY2hlbWEub2JqZWN0KHtcbiAgICAgICAgICBwYW5lbE5hbWU6IHNjaGVtYS5zdHJpbmcoKSxcbiAgICAgICAgICBhcHBsaWNhdGlvbklkOiBzY2hlbWEubWF5YmUoc2NoZW1hLnN0cmluZygpKSxcbiAgICAgICAgfSksXG4gICAgICB9LFxuICAgIH0sXG4gICAgYXN5bmMgKFxuICAgICAgY29udGV4dCxcbiAgICAgIHJlcXVlc3QsXG4gICAgICByZXNwb25zZVxuICAgICk6IFByb21pc2U8SU9wZW5TZWFyY2hEYXNoYm9hcmRzUmVzcG9uc2U8YW55IHwgUmVzcG9uc2VFcnJvcj4+ID0+IHtcbiAgICAgIGNvbnN0IG9wZW5zZWFyY2hOb3RlYm9va3NDbGllbnQ6IElMZWdhY3lTY29wZWRDbHVzdGVyQ2xpZW50ID0gY29udGV4dC5vYnNlcnZhYmlsaXR5X3BsdWdpbi5vYnNlcnZhYmlsaXR5Q2xpZW50LmFzU2NvcGVkKFxuICAgICAgICByZXF1ZXN0XG4gICAgICApO1xuICAgICAgdHJ5IHtcbiAgICAgICAgY29uc3QgbmV3UGFuZWxJZCA9IGF3YWl0IGN1c3RvbVBhbmVsQmFja2VuZC5jcmVhdGVOZXdQYW5lbChcbiAgICAgICAgICBvcGVuc2VhcmNoTm90ZWJvb2tzQ2xpZW50LFxuICAgICAgICAgIHJlcXVlc3QuYm9keS5wYW5lbE5hbWUsXG4gICAgICAgICAgcmVxdWVzdC5ib2R5LmFwcGxpY2F0aW9uSWQgfHwgJydcbiAgICAgICAgKTtcbiAgICAgICAgcmV0dXJuIHJlc3BvbnNlLm9rKHtcbiAgICAgICAgICBib2R5OiB7XG4gICAgICAgICAgICBtZXNzYWdlOiAnUGFuZWwgQ3JlYXRlZCcsXG4gICAgICAgICAgICBuZXdQYW5lbElkLFxuICAgICAgICAgIH0sXG4gICAgICAgIH0pO1xuICAgICAgfSBjYXRjaCAoZXJyb3I6IGFueSkge1xuICAgICAgICBjb25zb2xlLmVycm9yKCdJc3N1ZSBpbiBjcmVhdGluZyBuZXcgcGFuZWwnLCBlcnJvcik7XG4gICAgICAgIHJldHVybiByZXNwb25zZS5jdXN0b20oe1xuICAgICAgICAgIHN0YXR1c0NvZGU6IGVycm9yLnN0YXR1c0NvZGUgfHwgNTAwLFxuICAgICAgICAgIGJvZHk6IGVycm9yLm1lc3NhZ2UsXG4gICAgICAgIH0pO1xuICAgICAgfVxuICAgIH1cbiAgKTtcblxuICAvLyB1cGRhdGUgYW4gZXhpc3RpbmcgcGFuZWxcbiAgcm91dGVyLnBvc3QoXG4gICAge1xuICAgICAgcGF0aDogYCR7QVBJX1BSRUZJWH0vcGFuZWxzL3VwZGF0ZWAsXG4gICAgICB2YWxpZGF0ZToge1xuICAgICAgICBib2R5OiBzY2hlbWEub2JqZWN0KHtcbiAgICAgICAgICBwYW5lbElkOiBzY2hlbWEuc3RyaW5nKCksXG4gICAgICAgICAgcGFuZWw6IHNjaGVtYS5hbnkoKSxcbiAgICAgICAgfSksXG4gICAgICB9LFxuICAgIH0sXG4gICAgYXN5bmMgKFxuICAgICAgY29udGV4dCxcbiAgICAgIHJlcXVlc3QsXG4gICAgICByZXNwb25zZVxuICAgICk6IFByb21pc2U8SU9wZW5TZWFyY2hEYXNoYm9hcmRzUmVzcG9uc2U8YW55IHwgUmVzcG9uc2VFcnJvcj4+ID0+IHtcbiAgICAgIGNvbnN0IG9wZW5zZWFyY2hOb3RlYm9va3NDbGllbnQ6IElMZWdhY3lTY29wZWRDbHVzdGVyQ2xpZW50ID0gY29udGV4dC5vYnNlcnZhYmlsaXR5X3BsdWdpbi5vYnNlcnZhYmlsaXR5Q2xpZW50LmFzU2NvcGVkKFxuICAgICAgICByZXF1ZXN0XG4gICAgICApO1xuXG4gICAgICB0cnkge1xuICAgICAgICBjb25zdCByZXNwb25zZUJvZHkgPSBhd2FpdCBjdXN0b21QYW5lbEJhY2tlbmQudXBkYXRlUGFuZWwoXG4gICAgICAgICAgb3BlbnNlYXJjaE5vdGVib29rc0NsaWVudCxcbiAgICAgICAgICByZXF1ZXN0LmJvZHkucGFuZWxJZCxcbiAgICAgICAgICByZXF1ZXN0LmJvZHkucGFuZWxcbiAgICAgICAgKTtcbiAgICAgICAgcmV0dXJuIHJlc3BvbnNlLm9rKHtcbiAgICAgICAgICBib2R5OiB7XG4gICAgICAgICAgICBtZXNzYWdlOiAnUGFuZWwgVXBkYXRlZCcsXG4gICAgICAgICAgfSxcbiAgICAgICAgfSk7XG4gICAgICB9IGNhdGNoIChlcnJvcjogYW55KSB7XG4gICAgICAgIGNvbnNvbGUuZXJyb3IoJ0lzc3VlIGluIHVwZGF0aW5nIHBhbmVsJywgZXJyb3IpO1xuICAgICAgICByZXR1cm4gcmVzcG9uc2UuY3VzdG9tKHtcbiAgICAgICAgICBzdGF0dXNDb2RlOiBlcnJvci5zdGF0dXNDb2RlIHx8IDUwMCxcbiAgICAgICAgICBib2R5OiBlcnJvci5tZXNzYWdlLFxuICAgICAgICB9KTtcbiAgICAgIH1cbiAgICB9XG4gICk7XG5cbiAgLy8gcmVuYW1lIGFuIGV4aXN0aW5nIHBhbmVsXG4gIHJvdXRlci5wb3N0KFxuICAgIHtcbiAgICAgIHBhdGg6IGAke0FQSV9QUkVGSVh9L3BhbmVscy9yZW5hbWVgLFxuICAgICAgdmFsaWRhdGU6IHtcbiAgICAgICAgYm9keTogc2NoZW1hLm9iamVjdCh7XG4gICAgICAgICAgcGFuZWxJZDogc2NoZW1hLnN0cmluZygpLFxuICAgICAgICAgIHBhbmVsTmFtZTogc2NoZW1hLnN0cmluZygpLFxuICAgICAgICB9KSxcbiAgICAgIH0sXG4gICAgfSxcbiAgICBhc3luYyAoXG4gICAgICBjb250ZXh0LFxuICAgICAgcmVxdWVzdCxcbiAgICAgIHJlc3BvbnNlXG4gICAgKTogUHJvbWlzZTxJT3BlblNlYXJjaERhc2hib2FyZHNSZXNwb25zZTxhbnkgfCBSZXNwb25zZUVycm9yPj4gPT4ge1xuICAgICAgY29uc3Qgb3BlbnNlYXJjaE5vdGVib29rc0NsaWVudDogSUxlZ2FjeVNjb3BlZENsdXN0ZXJDbGllbnQgPSBjb250ZXh0Lm9ic2VydmFiaWxpdHlfcGx1Z2luLm9ic2VydmFiaWxpdHlDbGllbnQuYXNTY29wZWQoXG4gICAgICAgIHJlcXVlc3RcbiAgICAgICk7XG5cbiAgICAgIHRyeSB7XG4gICAgICAgIGNvbnN0IHJlc3BvbnNlQm9keSA9IGF3YWl0IGN1c3RvbVBhbmVsQmFja2VuZC5yZW5hbWVQYW5lbChcbiAgICAgICAgICBvcGVuc2VhcmNoTm90ZWJvb2tzQ2xpZW50LFxuICAgICAgICAgIHJlcXVlc3QuYm9keS5wYW5lbElkLFxuICAgICAgICAgIHJlcXVlc3QuYm9keS5wYW5lbE5hbWVcbiAgICAgICAgKTtcbiAgICAgICAgcmV0dXJuIHJlc3BvbnNlLm9rKHtcbiAgICAgICAgICBib2R5OiB7XG4gICAgICAgICAgICBtZXNzYWdlOiAnUGFuZWwgUmVuYW1lZCcsXG4gICAgICAgICAgfSxcbiAgICAgICAgfSk7XG4gICAgICB9IGNhdGNoIChlcnJvcjogYW55KSB7XG4gICAgICAgIGNvbnNvbGUuZXJyb3IoJ0lzc3VlIGluIHJlbmFtaW5nIHBhbmVsJywgZXJyb3IpO1xuICAgICAgICByZXR1cm4gcmVzcG9uc2UuY3VzdG9tKHtcbiAgICAgICAgICBzdGF0dXNDb2RlOiBlcnJvci5zdGF0dXNDb2RlIHx8IDUwMCxcbiAgICAgICAgICBib2R5OiBlcnJvci5tZXNzYWdlLFxuICAgICAgICB9KTtcbiAgICAgIH1cbiAgICB9XG4gICk7XG5cbiAgLy8gY2xvbmVzIGFuIGV4aXN0aW5nIHBhbmVsXG4gIC8vIHJldHVybnMgbmV3IHBhbmVsIElkXG4gIHJvdXRlci5wb3N0KFxuICAgIHtcbiAgICAgIHBhdGg6IGAke0FQSV9QUkVGSVh9L3BhbmVscy9jbG9uZWAsXG4gICAgICB2YWxpZGF0ZToge1xuICAgICAgICBib2R5OiBzY2hlbWEub2JqZWN0KHtcbiAgICAgICAgICBwYW5lbElkOiBzY2hlbWEuc3RyaW5nKCksXG4gICAgICAgICAgcGFuZWxOYW1lOiBzY2hlbWEuc3RyaW5nKCksXG4gICAgICAgIH0pLFxuICAgICAgfSxcbiAgICB9LFxuICAgIGFzeW5jIChcbiAgICAgIGNvbnRleHQsXG4gICAgICByZXF1ZXN0LFxuICAgICAgcmVzcG9uc2VcbiAgICApOiBQcm9taXNlPElPcGVuU2VhcmNoRGFzaGJvYXJkc1Jlc3BvbnNlPGFueSB8IFJlc3BvbnNlRXJyb3I+PiA9PiB7XG4gICAgICBjb25zdCBvcGVuc2VhcmNoTm90ZWJvb2tzQ2xpZW50OiBJTGVnYWN5U2NvcGVkQ2x1c3RlckNsaWVudCA9IGNvbnRleHQub2JzZXJ2YWJpbGl0eV9wbHVnaW4ub2JzZXJ2YWJpbGl0eUNsaWVudC5hc1Njb3BlZChcbiAgICAgICAgcmVxdWVzdFxuICAgICAgKTtcblxuICAgICAgdHJ5IHtcbiAgICAgICAgY29uc3QgY2xvbmVSZXNwb25zZSA9IGF3YWl0IGN1c3RvbVBhbmVsQmFja2VuZC5jbG9uZVBhbmVsKFxuICAgICAgICAgIG9wZW5zZWFyY2hOb3RlYm9va3NDbGllbnQsXG4gICAgICAgICAgcmVxdWVzdC5ib2R5LnBhbmVsSWQsXG4gICAgICAgICAgcmVxdWVzdC5ib2R5LnBhbmVsTmFtZVxuICAgICAgICApO1xuICAgICAgICByZXR1cm4gcmVzcG9uc2Uub2soe1xuICAgICAgICAgIGJvZHk6IHtcbiAgICAgICAgICAgIG1lc3NhZ2U6ICdQYW5lbCBDbG9uZWQnLFxuICAgICAgICAgICAgY2xvbmVQYW5lbElkOiBjbG9uZVJlc3BvbnNlLmNsb25lUGFuZWxJZCxcbiAgICAgICAgICAgIGRhdGVDcmVhdGVkOiBjbG9uZVJlc3BvbnNlLmRhdGVDcmVhdGVkLFxuICAgICAgICAgICAgZGF0ZU1vZGlmaWVkOiBjbG9uZVJlc3BvbnNlLmRhdGVNb2RpZmllZCxcbiAgICAgICAgICB9LFxuICAgICAgICB9KTtcbiAgICAgIH0gY2F0Y2ggKGVycm9yOiBhbnkpIHtcbiAgICAgICAgY29uc29sZS5lcnJvcignSXNzdWUgaW4gY2xvbmluZyBwYW5lbCcsIGVycm9yKTtcbiAgICAgICAgcmV0dXJuIHJlc3BvbnNlLmN1c3RvbSh7XG4gICAgICAgICAgc3RhdHVzQ29kZTogZXJyb3Iuc3RhdHVzQ29kZSB8fCA1MDAsXG4gICAgICAgICAgYm9keTogZXJyb3IubWVzc2FnZSxcbiAgICAgICAgfSk7XG4gICAgICB9XG4gICAgfVxuICApO1xuXG4gIC8vIGRlbGV0ZSBhbiBleGlzdGluZyBwYW5lbFxuICByb3V0ZXIuZGVsZXRlKFxuICAgIHtcbiAgICAgIHBhdGg6IGAke0FQSV9QUkVGSVh9L3BhbmVscy97cGFuZWxJZH1gLFxuICAgICAgdmFsaWRhdGU6IHtcbiAgICAgICAgcGFyYW1zOiBzY2hlbWEub2JqZWN0KHtcbiAgICAgICAgICBwYW5lbElkOiBzY2hlbWEuc3RyaW5nKCksXG4gICAgICAgIH0pLFxuICAgICAgfSxcbiAgICB9LFxuICAgIGFzeW5jIChcbiAgICAgIGNvbnRleHQsXG4gICAgICByZXF1ZXN0LFxuICAgICAgcmVzcG9uc2VcbiAgICApOiBQcm9taXNlPElPcGVuU2VhcmNoRGFzaGJvYXJkc1Jlc3BvbnNlPGFueSB8IFJlc3BvbnNlRXJyb3I+PiA9PiB7XG4gICAgICBjb25zdCBvcGVuc2VhcmNoTm90ZWJvb2tzQ2xpZW50OiBJTGVnYWN5U2NvcGVkQ2x1c3RlckNsaWVudCA9IGNvbnRleHQub2JzZXJ2YWJpbGl0eV9wbHVnaW4ub2JzZXJ2YWJpbGl0eUNsaWVudC5hc1Njb3BlZChcbiAgICAgICAgcmVxdWVzdFxuICAgICAgKTtcblxuICAgICAgdHJ5IHtcbiAgICAgICAgY29uc3QgZGVsZXRlUmVzcG9uc2UgPSBhd2FpdCBjdXN0b21QYW5lbEJhY2tlbmQuZGVsZXRlUGFuZWwoXG4gICAgICAgICAgb3BlbnNlYXJjaE5vdGVib29rc0NsaWVudCxcbiAgICAgICAgICByZXF1ZXN0LnBhcmFtcy5wYW5lbElkXG4gICAgICAgICk7XG4gICAgICAgIHJldHVybiByZXNwb25zZS5vayh7XG4gICAgICAgICAgYm9keToge1xuICAgICAgICAgICAgbWVzc2FnZTogJ1BhbmVsIERlbGV0ZWQnLFxuICAgICAgICAgIH0sXG4gICAgICAgIH0pO1xuICAgICAgfSBjYXRjaCAoZXJyb3I6IGFueSkge1xuICAgICAgICBjb25zb2xlLmVycm9yKCdJc3N1ZSBpbiBkZWxldGluZyBwYW5lbCcsIGVycm9yKTtcbiAgICAgICAgcmV0dXJuIHJlc3BvbnNlLmN1c3RvbSh7XG4gICAgICAgICAgc3RhdHVzQ29kZTogZXJyb3Iuc3RhdHVzQ29kZSB8fCA1MDAsXG4gICAgICAgICAgYm9keTogZXJyb3IubWVzc2FnZSxcbiAgICAgICAgfSk7XG4gICAgICB9XG4gICAgfVxuICApO1xuXG4gIC8vIGRlbGV0ZSBhbiBleGlzdGluZyBwYW5lbChzKVxuICByb3V0ZXIuZGVsZXRlKFxuICAgIHtcbiAgICAgIHBhdGg6IGAke0FQSV9QUkVGSVh9L3BhbmVsTGlzdC97cGFuZWxJZExpc3R9YCxcbiAgICAgIHZhbGlkYXRlOiB7XG4gICAgICAgIHBhcmFtczogc2NoZW1hLm9iamVjdCh7XG4gICAgICAgICAgcGFuZWxJZExpc3Q6IHNjaGVtYS5zdHJpbmcoKSxcbiAgICAgICAgfSksXG4gICAgICB9LFxuICAgIH0sXG4gICAgYXN5bmMgKFxuICAgICAgY29udGV4dCxcbiAgICAgIHJlcXVlc3QsXG4gICAgICByZXNwb25zZVxuICAgICk6IFByb21pc2U8SU9wZW5TZWFyY2hEYXNoYm9hcmRzUmVzcG9uc2U8YW55IHwgUmVzcG9uc2VFcnJvcj4+ID0+IHtcbiAgICAgIGNvbnN0IG9wZW5zZWFyY2hOb3RlYm9va3NDbGllbnQ6IElMZWdhY3lTY29wZWRDbHVzdGVyQ2xpZW50ID0gY29udGV4dC5vYnNlcnZhYmlsaXR5X3BsdWdpbi5vYnNlcnZhYmlsaXR5Q2xpZW50LmFzU2NvcGVkKFxuICAgICAgICByZXF1ZXN0XG4gICAgICApO1xuXG4gICAgICB0cnkge1xuICAgICAgICBjb25zdCBkZWxldGVSZXNwb25zZSA9IGF3YWl0IGN1c3RvbVBhbmVsQmFja2VuZC5kZWxldGVQYW5lbExpc3QoXG4gICAgICAgICAgb3BlbnNlYXJjaE5vdGVib29rc0NsaWVudCxcbiAgICAgICAgICByZXF1ZXN0LnBhcmFtcy5wYW5lbElkTGlzdFxuICAgICAgICApO1xuICAgICAgICByZXR1cm4gcmVzcG9uc2Uub2soe1xuICAgICAgICAgIGJvZHk6IHtcbiAgICAgICAgICAgIG1lc3NhZ2U6ICdQYW5lbCBEZWxldGVkJyxcbiAgICAgICAgICB9LFxuICAgICAgICB9KTtcbiAgICAgIH0gY2F0Y2ggKGVycm9yOiBhbnkpIHtcbiAgICAgICAgY29uc29sZS5lcnJvcignSXNzdWUgaW4gZGVsZXRpbmcgcGFuZWwnLCBlcnJvcik7XG4gICAgICAgIHJldHVybiByZXNwb25zZS5jdXN0b20oe1xuICAgICAgICAgIHN0YXR1c0NvZGU6IGVycm9yLnN0YXR1c0NvZGUgfHwgNTAwLFxuICAgICAgICAgIGJvZHk6IGVycm9yLm1lc3NhZ2UsXG4gICAgICAgIH0pO1xuICAgICAgfVxuICAgIH1cbiAgKTtcblxuICAvLyByZXBsYWNlcyB0aGUgcHBsIHF1ZXJ5IGZpbHRlciBpbiBwYW5lbFxuICByb3V0ZXIucG9zdChcbiAgICB7XG4gICAgICBwYXRoOiBgJHtBUElfUFJFRklYfS9wYW5lbHMvZmlsdGVyYCxcbiAgICAgIHZhbGlkYXRlOiB7XG4gICAgICAgIGJvZHk6IHNjaGVtYS5vYmplY3Qoe1xuICAgICAgICAgIHBhbmVsSWQ6IHNjaGVtYS5zdHJpbmcoKSxcbiAgICAgICAgICBxdWVyeTogc2NoZW1hLnN0cmluZygpLFxuICAgICAgICAgIGxhbmd1YWdlOiBzY2hlbWEuc3RyaW5nKCksXG4gICAgICAgICAgdG86IHNjaGVtYS5zdHJpbmcoKSxcbiAgICAgICAgICBmcm9tOiBzY2hlbWEuc3RyaW5nKCksXG4gICAgICAgIH0pLFxuICAgICAgfSxcbiAgICB9LFxuICAgIGFzeW5jIChcbiAgICAgIGNvbnRleHQsXG4gICAgICByZXF1ZXN0LFxuICAgICAgcmVzcG9uc2VcbiAgICApOiBQcm9taXNlPElPcGVuU2VhcmNoRGFzaGJvYXJkc1Jlc3BvbnNlPGFueSB8IFJlc3BvbnNlRXJyb3I+PiA9PiB7XG4gICAgICBjb25zdCBvcGVuc2VhcmNoTm90ZWJvb2tzQ2xpZW50OiBJTGVnYWN5U2NvcGVkQ2x1c3RlckNsaWVudCA9IGNvbnRleHQub2JzZXJ2YWJpbGl0eV9wbHVnaW4ub2JzZXJ2YWJpbGl0eUNsaWVudC5hc1Njb3BlZChcbiAgICAgICAgcmVxdWVzdFxuICAgICAgKTtcblxuICAgICAgdHJ5IHtcbiAgICAgICAgY29uc3QgcGFuZWxGaWx0ZXJSZXNwb25zZSA9IGF3YWl0IGN1c3RvbVBhbmVsQmFja2VuZC5hZGRQYW5lbEZpbHRlcihcbiAgICAgICAgICBvcGVuc2VhcmNoTm90ZWJvb2tzQ2xpZW50LFxuICAgICAgICAgIHJlcXVlc3QuYm9keS5wYW5lbElkLFxuICAgICAgICAgIHJlcXVlc3QuYm9keS5xdWVyeSxcbiAgICAgICAgICByZXF1ZXN0LmJvZHkubGFuZ3VhZ2UsXG4gICAgICAgICAgcmVxdWVzdC5ib2R5LnRvLFxuICAgICAgICAgIHJlcXVlc3QuYm9keS5mcm9tXG4gICAgICAgICk7XG4gICAgICAgIHJldHVybiByZXNwb25zZS5vayh7XG4gICAgICAgICAgYm9keToge1xuICAgICAgICAgICAgbWVzc2FnZTogJ1BhbmVsIFBQTCBGaWx0ZXIgQ2hhbmdlZCcsXG4gICAgICAgICAgfSxcbiAgICAgICAgfSk7XG4gICAgICB9IGNhdGNoIChlcnJvcjogYW55KSB7XG4gICAgICAgIGNvbnNvbGUuZXJyb3IoJ0lzc3VlIGluIGFkZGluZyBxdWVyeSBmaWx0ZXInLCBlcnJvcik7XG4gICAgICAgIHJldHVybiByZXNwb25zZS5jdXN0b20oe1xuICAgICAgICAgIHN0YXR1c0NvZGU6IGVycm9yLnN0YXR1c0NvZGUgfHwgNTAwLFxuICAgICAgICAgIGJvZHk6IGVycm9yLm1lc3NhZ2UsXG4gICAgICAgIH0pO1xuICAgICAgfVxuICAgIH1cbiAgKTtcblxuICAvLyBBZGQgU2FtcGxlIFBhbmVsc1xuICByb3V0ZXIucG9zdChcbiAgICB7XG4gICAgICBwYXRoOiBgJHtBUElfUFJFRklYfS9wYW5lbHMvYWRkU2FtcGxlUGFuZWxzYCxcbiAgICAgIHZhbGlkYXRlOiB7XG4gICAgICAgIGJvZHk6IHNjaGVtYS5vYmplY3Qoe1xuICAgICAgICAgIHNhdmVkVmlzdWFsaXphdGlvbklkczogc2NoZW1hLmFycmF5T2Yoc2NoZW1hLnN0cmluZygpKSxcbiAgICAgICAgfSksXG4gICAgICB9LFxuICAgIH0sXG4gICAgYXN5bmMgKFxuICAgICAgY29udGV4dCxcbiAgICAgIHJlcXVlc3QsXG4gICAgICByZXNwb25zZVxuICAgICk6IFByb21pc2U8SU9wZW5TZWFyY2hEYXNoYm9hcmRzUmVzcG9uc2U8YW55IHwgUmVzcG9uc2VFcnJvcj4+ID0+IHtcbiAgICAgIGNvbnN0IG9wZW5zZWFyY2hOb3RlYm9va3NDbGllbnQ6IElMZWdhY3lTY29wZWRDbHVzdGVyQ2xpZW50ID0gY29udGV4dC5vYnNlcnZhYmlsaXR5X3BsdWdpbi5vYnNlcnZhYmlsaXR5Q2xpZW50LmFzU2NvcGVkKFxuICAgICAgICByZXF1ZXN0XG4gICAgICApO1xuXG4gICAgICB0cnkge1xuICAgICAgICBjb25zdCBwYW5lbHNEYXRhID0gYXdhaXQgY3VzdG9tUGFuZWxCYWNrZW5kLmFkZFNhbXBsZVBhbmVscyhcbiAgICAgICAgICBvcGVuc2VhcmNoTm90ZWJvb2tzQ2xpZW50LFxuICAgICAgICAgIHJlcXVlc3QuYm9keS5zYXZlZFZpc3VhbGl6YXRpb25JZHNcbiAgICAgICAgKTtcbiAgICAgICAgcmV0dXJuIHJlc3BvbnNlLm9rKHtcbiAgICAgICAgICBib2R5OiB7XG4gICAgICAgICAgICBkZW1vUGFuZWxzRGF0YTogcGFuZWxzRGF0YSxcbiAgICAgICAgICB9LFxuICAgICAgICB9KTtcbiAgICAgIH0gY2F0Y2ggKGVycm9yOiBhbnkpIHtcbiAgICAgICAgY29uc29sZS5lcnJvcignSXNzdWUgaW4gZmV0Y2hpbmcgcGFuZWwgbGlzdDonLCBlcnJvcik7XG4gICAgICAgIHJldHVybiByZXNwb25zZS5jdXN0b20oe1xuICAgICAgICAgIHN0YXR1c0NvZGU6IGVycm9yLnN0YXR1c0NvZGUgfHwgNTAwLFxuICAgICAgICAgIGJvZHk6IGVycm9yLm1lc3NhZ2UsXG4gICAgICAgIH0pO1xuICAgICAgfVxuICAgIH1cbiAgKTtcbn1cbiJdLCJtYXBwaW5ncyI6Ijs7Ozs7O0FBS0EsSUFBQUEsYUFBQSxHQUFBQyxPQUFBO0FBQ0EsSUFBQUMscUJBQUEsR0FBQUQsT0FBQTtBQU9BLElBQUFFLGNBQUEsR0FBQUYsT0FBQTtBQWJBO0FBQ0E7QUFDQTtBQUNBOztBQVlPLFNBQVNHLFlBQVlBLENBQUNDLE1BQWUsRUFBRTtFQUM1QyxNQUFNQyxrQkFBa0IsR0FBRyxJQUFJQyx5Q0FBbUIsQ0FBQyxDQUFDO0VBQ3BEO0VBQ0FGLE1BQU0sQ0FBQ0csR0FBRyxDQUNSO0lBQ0VDLElBQUksRUFBRyxHQUFFQyx1Q0FBVyxTQUFRO0lBQzVCQyxRQUFRLEVBQUUsQ0FBQztFQUNiLENBQUMsRUFDRCxPQUNFQyxPQUFPLEVBQ1BDLE9BQU8sRUFDUEMsUUFBUSxLQUN3RDtJQUNoRSxNQUFNQyx5QkFBcUQsR0FBR0gsT0FBTyxDQUFDSSxvQkFBb0IsQ0FBQ0MsbUJBQW1CLENBQUNDLFFBQVEsQ0FDckhMLE9BQ0YsQ0FBQztJQUVELElBQUk7TUFDRixNQUFNTSxVQUFVLEdBQUcsTUFBTWIsa0JBQWtCLENBQUNjLGFBQWEsQ0FBQ0wseUJBQXlCLENBQUM7TUFDcEYsT0FBT0QsUUFBUSxDQUFDTyxFQUFFLENBQUM7UUFDakJDLElBQUksRUFBRTtVQUNKQyxNQUFNLEVBQUVKO1FBQ1Y7TUFDRixDQUFDLENBQUM7SUFDSixDQUFDLENBQUMsT0FBT0ssS0FBVSxFQUFFO01BQ25CQyxPQUFPLENBQUNELEtBQUssQ0FBQywrQkFBK0IsRUFBRUEsS0FBSyxDQUFDO01BQ3JELE9BQU9WLFFBQVEsQ0FBQ1ksTUFBTSxDQUFDO1FBQ3JCQyxVQUFVLEVBQUVILEtBQUssQ0FBQ0csVUFBVSxJQUFJLEdBQUc7UUFDbkNMLElBQUksRUFBRUUsS0FBSyxDQUFDSTtNQUNkLENBQUMsQ0FBQztJQUNKO0VBQ0YsQ0FDRixDQUFDOztFQUVEO0VBQ0F2QixNQUFNLENBQUNHLEdBQUcsQ0FDUjtJQUNFQyxJQUFJLEVBQUcsR0FBRUMsdUNBQVcsbUJBQWtCO0lBQ3RDQyxRQUFRLEVBQUU7TUFDUmtCLE1BQU0sRUFBRUMsb0JBQU0sQ0FBQ0MsTUFBTSxDQUFDO1FBQ3BCQyxPQUFPLEVBQUVGLG9CQUFNLENBQUNHLE1BQU0sQ0FBQztNQUN6QixDQUFDO0lBQ0g7RUFDRixDQUFDLEVBQ0QsT0FDRXJCLE9BQU8sRUFDUEMsT0FBTyxFQUNQQyxRQUFRLEtBQ3dEO0lBQ2hFLE1BQU1DLHlCQUFxRCxHQUFHSCxPQUFPLENBQUNJLG9CQUFvQixDQUFDQyxtQkFBbUIsQ0FBQ0MsUUFBUSxDQUNySEwsT0FDRixDQUFDO0lBRUQsSUFBSTtNQUNGLE1BQU1xQixXQUFXLEdBQUcsTUFBTTVCLGtCQUFrQixDQUFDNkIsUUFBUSxDQUNuRHBCLHlCQUF5QixFQUN6QkYsT0FBTyxDQUFDZ0IsTUFBTSxDQUFDRyxPQUNqQixDQUFDO01BQ0QsT0FBT2xCLFFBQVEsQ0FBQ08sRUFBRSxDQUFDO1FBQ2pCQyxJQUFJLEVBQUVZO01BQ1IsQ0FBQyxDQUFDO0lBQ0osQ0FBQyxDQUFDLE9BQU9WLEtBQVUsRUFBRTtNQUNuQkMsT0FBTyxDQUFDRCxLQUFLLENBQUMsMEJBQTBCLEVBQUVBLEtBQUssQ0FBQztNQUNoRCxPQUFPVixRQUFRLENBQUNZLE1BQU0sQ0FBQztRQUNyQkMsVUFBVSxFQUFFSCxLQUFLLENBQUNHLFVBQVUsSUFBSSxHQUFHO1FBQ25DTCxJQUFJLEVBQUVFLEtBQUssQ0FBQ0k7TUFDZCxDQUFDLENBQUM7SUFDSjtFQUNGLENBQ0YsQ0FBQzs7RUFFRDtFQUNBdkIsTUFBTSxDQUFDK0IsSUFBSSxDQUNUO0lBQ0UzQixJQUFJLEVBQUcsR0FBRUMsdUNBQVcsU0FBUTtJQUM1QkMsUUFBUSxFQUFFO01BQ1JXLElBQUksRUFBRVEsb0JBQU0sQ0FBQ0MsTUFBTSxDQUFDO1FBQ2xCTSxTQUFTLEVBQUVQLG9CQUFNLENBQUNHLE1BQU0sQ0FBQyxDQUFDO1FBQzFCSyxhQUFhLEVBQUVSLG9CQUFNLENBQUNTLEtBQUssQ0FBQ1Qsb0JBQU0sQ0FBQ0csTUFBTSxDQUFDLENBQUM7TUFDN0MsQ0FBQztJQUNIO0VBQ0YsQ0FBQyxFQUNELE9BQ0VyQixPQUFPLEVBQ1BDLE9BQU8sRUFDUEMsUUFBUSxLQUN3RDtJQUNoRSxNQUFNQyx5QkFBcUQsR0FBR0gsT0FBTyxDQUFDSSxvQkFBb0IsQ0FBQ0MsbUJBQW1CLENBQUNDLFFBQVEsQ0FDckhMLE9BQ0YsQ0FBQztJQUNELElBQUk7TUFDRixNQUFNMkIsVUFBVSxHQUFHLE1BQU1sQyxrQkFBa0IsQ0FBQ21DLGNBQWMsQ0FDeEQxQix5QkFBeUIsRUFDekJGLE9BQU8sQ0FBQ1MsSUFBSSxDQUFDZSxTQUFTLEVBQ3RCeEIsT0FBTyxDQUFDUyxJQUFJLENBQUNnQixhQUFhLElBQUksRUFDaEMsQ0FBQztNQUNELE9BQU94QixRQUFRLENBQUNPLEVBQUUsQ0FBQztRQUNqQkMsSUFBSSxFQUFFO1VBQ0pNLE9BQU8sRUFBRSxlQUFlO1VBQ3hCWTtRQUNGO01BQ0YsQ0FBQyxDQUFDO0lBQ0osQ0FBQyxDQUFDLE9BQU9oQixLQUFVLEVBQUU7TUFDbkJDLE9BQU8sQ0FBQ0QsS0FBSyxDQUFDLDZCQUE2QixFQUFFQSxLQUFLLENBQUM7TUFDbkQsT0FBT1YsUUFBUSxDQUFDWSxNQUFNLENBQUM7UUFDckJDLFVBQVUsRUFBRUgsS0FBSyxDQUFDRyxVQUFVLElBQUksR0FBRztRQUNuQ0wsSUFBSSxFQUFFRSxLQUFLLENBQUNJO01BQ2QsQ0FBQyxDQUFDO0lBQ0o7RUFDRixDQUNGLENBQUM7O0VBRUQ7RUFDQXZCLE1BQU0sQ0FBQytCLElBQUksQ0FDVDtJQUNFM0IsSUFBSSxFQUFHLEdBQUVDLHVDQUFXLGdCQUFlO0lBQ25DQyxRQUFRLEVBQUU7TUFDUlcsSUFBSSxFQUFFUSxvQkFBTSxDQUFDQyxNQUFNLENBQUM7UUFDbEJDLE9BQU8sRUFBRUYsb0JBQU0sQ0FBQ0csTUFBTSxDQUFDLENBQUM7UUFDeEJTLEtBQUssRUFBRVosb0JBQU0sQ0FBQ2EsR0FBRyxDQUFDO01BQ3BCLENBQUM7SUFDSDtFQUNGLENBQUMsRUFDRCxPQUNFL0IsT0FBTyxFQUNQQyxPQUFPLEVBQ1BDLFFBQVEsS0FDd0Q7SUFDaEUsTUFBTUMseUJBQXFELEdBQUdILE9BQU8sQ0FBQ0ksb0JBQW9CLENBQUNDLG1CQUFtQixDQUFDQyxRQUFRLENBQ3JITCxPQUNGLENBQUM7SUFFRCxJQUFJO01BQ0YsTUFBTStCLFlBQVksR0FBRyxNQUFNdEMsa0JBQWtCLENBQUN1QyxXQUFXLENBQ3ZEOUIseUJBQXlCLEVBQ3pCRixPQUFPLENBQUNTLElBQUksQ0FBQ1UsT0FBTyxFQUNwQm5CLE9BQU8sQ0FBQ1MsSUFBSSxDQUFDb0IsS0FDZixDQUFDO01BQ0QsT0FBTzVCLFFBQVEsQ0FBQ08sRUFBRSxDQUFDO1FBQ2pCQyxJQUFJLEVBQUU7VUFDSk0sT0FBTyxFQUFFO1FBQ1g7TUFDRixDQUFDLENBQUM7SUFDSixDQUFDLENBQUMsT0FBT0osS0FBVSxFQUFFO01BQ25CQyxPQUFPLENBQUNELEtBQUssQ0FBQyx5QkFBeUIsRUFBRUEsS0FBSyxDQUFDO01BQy9DLE9BQU9WLFFBQVEsQ0FBQ1ksTUFBTSxDQUFDO1FBQ3JCQyxVQUFVLEVBQUVILEtBQUssQ0FBQ0csVUFBVSxJQUFJLEdBQUc7UUFDbkNMLElBQUksRUFBRUUsS0FBSyxDQUFDSTtNQUNkLENBQUMsQ0FBQztJQUNKO0VBQ0YsQ0FDRixDQUFDOztFQUVEO0VBQ0F2QixNQUFNLENBQUMrQixJQUFJLENBQ1Q7SUFDRTNCLElBQUksRUFBRyxHQUFFQyx1Q0FBVyxnQkFBZTtJQUNuQ0MsUUFBUSxFQUFFO01BQ1JXLElBQUksRUFBRVEsb0JBQU0sQ0FBQ0MsTUFBTSxDQUFDO1FBQ2xCQyxPQUFPLEVBQUVGLG9CQUFNLENBQUNHLE1BQU0sQ0FBQyxDQUFDO1FBQ3hCSSxTQUFTLEVBQUVQLG9CQUFNLENBQUNHLE1BQU0sQ0FBQztNQUMzQixDQUFDO0lBQ0g7RUFDRixDQUFDLEVBQ0QsT0FDRXJCLE9BQU8sRUFDUEMsT0FBTyxFQUNQQyxRQUFRLEtBQ3dEO0lBQ2hFLE1BQU1DLHlCQUFxRCxHQUFHSCxPQUFPLENBQUNJLG9CQUFvQixDQUFDQyxtQkFBbUIsQ0FBQ0MsUUFBUSxDQUNySEwsT0FDRixDQUFDO0lBRUQsSUFBSTtNQUNGLE1BQU0rQixZQUFZLEdBQUcsTUFBTXRDLGtCQUFrQixDQUFDd0MsV0FBVyxDQUN2RC9CLHlCQUF5QixFQUN6QkYsT0FBTyxDQUFDUyxJQUFJLENBQUNVLE9BQU8sRUFDcEJuQixPQUFPLENBQUNTLElBQUksQ0FBQ2UsU0FDZixDQUFDO01BQ0QsT0FBT3ZCLFFBQVEsQ0FBQ08sRUFBRSxDQUFDO1FBQ2pCQyxJQUFJLEVBQUU7VUFDSk0sT0FBTyxFQUFFO1FBQ1g7TUFDRixDQUFDLENBQUM7SUFDSixDQUFDLENBQUMsT0FBT0osS0FBVSxFQUFFO01BQ25CQyxPQUFPLENBQUNELEtBQUssQ0FBQyx5QkFBeUIsRUFBRUEsS0FBSyxDQUFDO01BQy9DLE9BQU9WLFFBQVEsQ0FBQ1ksTUFBTSxDQUFDO1FBQ3JCQyxVQUFVLEVBQUVILEtBQUssQ0FBQ0csVUFBVSxJQUFJLEdBQUc7UUFDbkNMLElBQUksRUFBRUUsS0FBSyxDQUFDSTtNQUNkLENBQUMsQ0FBQztJQUNKO0VBQ0YsQ0FDRixDQUFDOztFQUVEO0VBQ0E7RUFDQXZCLE1BQU0sQ0FBQytCLElBQUksQ0FDVDtJQUNFM0IsSUFBSSxFQUFHLEdBQUVDLHVDQUFXLGVBQWM7SUFDbENDLFFBQVEsRUFBRTtNQUNSVyxJQUFJLEVBQUVRLG9CQUFNLENBQUNDLE1BQU0sQ0FBQztRQUNsQkMsT0FBTyxFQUFFRixvQkFBTSxDQUFDRyxNQUFNLENBQUMsQ0FBQztRQUN4QkksU0FBUyxFQUFFUCxvQkFBTSxDQUFDRyxNQUFNLENBQUM7TUFDM0IsQ0FBQztJQUNIO0VBQ0YsQ0FBQyxFQUNELE9BQ0VyQixPQUFPLEVBQ1BDLE9BQU8sRUFDUEMsUUFBUSxLQUN3RDtJQUNoRSxNQUFNQyx5QkFBcUQsR0FBR0gsT0FBTyxDQUFDSSxvQkFBb0IsQ0FBQ0MsbUJBQW1CLENBQUNDLFFBQVEsQ0FDckhMLE9BQ0YsQ0FBQztJQUVELElBQUk7TUFDRixNQUFNa0MsYUFBYSxHQUFHLE1BQU16QyxrQkFBa0IsQ0FBQzBDLFVBQVUsQ0FDdkRqQyx5QkFBeUIsRUFDekJGLE9BQU8sQ0FBQ1MsSUFBSSxDQUFDVSxPQUFPLEVBQ3BCbkIsT0FBTyxDQUFDUyxJQUFJLENBQUNlLFNBQ2YsQ0FBQztNQUNELE9BQU92QixRQUFRLENBQUNPLEVBQUUsQ0FBQztRQUNqQkMsSUFBSSxFQUFFO1VBQ0pNLE9BQU8sRUFBRSxjQUFjO1VBQ3ZCcUIsWUFBWSxFQUFFRixhQUFhLENBQUNFLFlBQVk7VUFDeENDLFdBQVcsRUFBRUgsYUFBYSxDQUFDRyxXQUFXO1VBQ3RDQyxZQUFZLEVBQUVKLGFBQWEsQ0FBQ0k7UUFDOUI7TUFDRixDQUFDLENBQUM7SUFDSixDQUFDLENBQUMsT0FBTzNCLEtBQVUsRUFBRTtNQUNuQkMsT0FBTyxDQUFDRCxLQUFLLENBQUMsd0JBQXdCLEVBQUVBLEtBQUssQ0FBQztNQUM5QyxPQUFPVixRQUFRLENBQUNZLE1BQU0sQ0FBQztRQUNyQkMsVUFBVSxFQUFFSCxLQUFLLENBQUNHLFVBQVUsSUFBSSxHQUFHO1FBQ25DTCxJQUFJLEVBQUVFLEtBQUssQ0FBQ0k7TUFDZCxDQUFDLENBQUM7SUFDSjtFQUNGLENBQ0YsQ0FBQzs7RUFFRDtFQUNBdkIsTUFBTSxDQUFDK0MsTUFBTSxDQUNYO0lBQ0UzQyxJQUFJLEVBQUcsR0FBRUMsdUNBQVcsbUJBQWtCO0lBQ3RDQyxRQUFRLEVBQUU7TUFDUmtCLE1BQU0sRUFBRUMsb0JBQU0sQ0FBQ0MsTUFBTSxDQUFDO1FBQ3BCQyxPQUFPLEVBQUVGLG9CQUFNLENBQUNHLE1BQU0sQ0FBQztNQUN6QixDQUFDO0lBQ0g7RUFDRixDQUFDLEVBQ0QsT0FDRXJCLE9BQU8sRUFDUEMsT0FBTyxFQUNQQyxRQUFRLEtBQ3dEO0lBQ2hFLE1BQU1DLHlCQUFxRCxHQUFHSCxPQUFPLENBQUNJLG9CQUFvQixDQUFDQyxtQkFBbUIsQ0FBQ0MsUUFBUSxDQUNySEwsT0FDRixDQUFDO0lBRUQsSUFBSTtNQUNGLE1BQU13QyxjQUFjLEdBQUcsTUFBTS9DLGtCQUFrQixDQUFDZ0QsV0FBVyxDQUN6RHZDLHlCQUF5QixFQUN6QkYsT0FBTyxDQUFDZ0IsTUFBTSxDQUFDRyxPQUNqQixDQUFDO01BQ0QsT0FBT2xCLFFBQVEsQ0FBQ08sRUFBRSxDQUFDO1FBQ2pCQyxJQUFJLEVBQUU7VUFDSk0sT0FBTyxFQUFFO1FBQ1g7TUFDRixDQUFDLENBQUM7SUFDSixDQUFDLENBQUMsT0FBT0osS0FBVSxFQUFFO01BQ25CQyxPQUFPLENBQUNELEtBQUssQ0FBQyx5QkFBeUIsRUFBRUEsS0FBSyxDQUFDO01BQy9DLE9BQU9WLFFBQVEsQ0FBQ1ksTUFBTSxDQUFDO1FBQ3JCQyxVQUFVLEVBQUVILEtBQUssQ0FBQ0csVUFBVSxJQUFJLEdBQUc7UUFDbkNMLElBQUksRUFBRUUsS0FBSyxDQUFDSTtNQUNkLENBQUMsQ0FBQztJQUNKO0VBQ0YsQ0FDRixDQUFDOztFQUVEO0VBQ0F2QixNQUFNLENBQUMrQyxNQUFNLENBQ1g7SUFDRTNDLElBQUksRUFBRyxHQUFFQyx1Q0FBVywwQkFBeUI7SUFDN0NDLFFBQVEsRUFBRTtNQUNSa0IsTUFBTSxFQUFFQyxvQkFBTSxDQUFDQyxNQUFNLENBQUM7UUFDcEJ3QixXQUFXLEVBQUV6QixvQkFBTSxDQUFDRyxNQUFNLENBQUM7TUFDN0IsQ0FBQztJQUNIO0VBQ0YsQ0FBQyxFQUNELE9BQ0VyQixPQUFPLEVBQ1BDLE9BQU8sRUFDUEMsUUFBUSxLQUN3RDtJQUNoRSxNQUFNQyx5QkFBcUQsR0FBR0gsT0FBTyxDQUFDSSxvQkFBb0IsQ0FBQ0MsbUJBQW1CLENBQUNDLFFBQVEsQ0FDckhMLE9BQ0YsQ0FBQztJQUVELElBQUk7TUFDRixNQUFNd0MsY0FBYyxHQUFHLE1BQU0vQyxrQkFBa0IsQ0FBQ2tELGVBQWUsQ0FDN0R6Qyx5QkFBeUIsRUFDekJGLE9BQU8sQ0FBQ2dCLE1BQU0sQ0FBQzBCLFdBQ2pCLENBQUM7TUFDRCxPQUFPekMsUUFBUSxDQUFDTyxFQUFFLENBQUM7UUFDakJDLElBQUksRUFBRTtVQUNKTSxPQUFPLEVBQUU7UUFDWDtNQUNGLENBQUMsQ0FBQztJQUNKLENBQUMsQ0FBQyxPQUFPSixLQUFVLEVBQUU7TUFDbkJDLE9BQU8sQ0FBQ0QsS0FBSyxDQUFDLHlCQUF5QixFQUFFQSxLQUFLLENBQUM7TUFDL0MsT0FBT1YsUUFBUSxDQUFDWSxNQUFNLENBQUM7UUFDckJDLFVBQVUsRUFBRUgsS0FBSyxDQUFDRyxVQUFVLElBQUksR0FBRztRQUNuQ0wsSUFBSSxFQUFFRSxLQUFLLENBQUNJO01BQ2QsQ0FBQyxDQUFDO0lBQ0o7RUFDRixDQUNGLENBQUM7O0VBRUQ7RUFDQXZCLE1BQU0sQ0FBQytCLElBQUksQ0FDVDtJQUNFM0IsSUFBSSxFQUFHLEdBQUVDLHVDQUFXLGdCQUFlO0lBQ25DQyxRQUFRLEVBQUU7TUFDUlcsSUFBSSxFQUFFUSxvQkFBTSxDQUFDQyxNQUFNLENBQUM7UUFDbEJDLE9BQU8sRUFBRUYsb0JBQU0sQ0FBQ0csTUFBTSxDQUFDLENBQUM7UUFDeEJ3QixLQUFLLEVBQUUzQixvQkFBTSxDQUFDRyxNQUFNLENBQUMsQ0FBQztRQUN0QnlCLFFBQVEsRUFBRTVCLG9CQUFNLENBQUNHLE1BQU0sQ0FBQyxDQUFDO1FBQ3pCMEIsRUFBRSxFQUFFN0Isb0JBQU0sQ0FBQ0csTUFBTSxDQUFDLENBQUM7UUFDbkIyQixJQUFJLEVBQUU5QixvQkFBTSxDQUFDRyxNQUFNLENBQUM7TUFDdEIsQ0FBQztJQUNIO0VBQ0YsQ0FBQyxFQUNELE9BQ0VyQixPQUFPLEVBQ1BDLE9BQU8sRUFDUEMsUUFBUSxLQUN3RDtJQUNoRSxNQUFNQyx5QkFBcUQsR0FBR0gsT0FBTyxDQUFDSSxvQkFBb0IsQ0FBQ0MsbUJBQW1CLENBQUNDLFFBQVEsQ0FDckhMLE9BQ0YsQ0FBQztJQUVELElBQUk7TUFDRixNQUFNZ0QsbUJBQW1CLEdBQUcsTUFBTXZELGtCQUFrQixDQUFDd0QsY0FBYyxDQUNqRS9DLHlCQUF5QixFQUN6QkYsT0FBTyxDQUFDUyxJQUFJLENBQUNVLE9BQU8sRUFDcEJuQixPQUFPLENBQUNTLElBQUksQ0FBQ21DLEtBQUssRUFDbEI1QyxPQUFPLENBQUNTLElBQUksQ0FBQ29DLFFBQVEsRUFDckI3QyxPQUFPLENBQUNTLElBQUksQ0FBQ3FDLEVBQUUsRUFDZjlDLE9BQU8sQ0FBQ1MsSUFBSSxDQUFDc0MsSUFDZixDQUFDO01BQ0QsT0FBTzlDLFFBQVEsQ0FBQ08sRUFBRSxDQUFDO1FBQ2pCQyxJQUFJLEVBQUU7VUFDSk0sT0FBTyxFQUFFO1FBQ1g7TUFDRixDQUFDLENBQUM7SUFDSixDQUFDLENBQUMsT0FBT0osS0FBVSxFQUFFO01BQ25CQyxPQUFPLENBQUNELEtBQUssQ0FBQyw4QkFBOEIsRUFBRUEsS0FBSyxDQUFDO01BQ3BELE9BQU9WLFFBQVEsQ0FBQ1ksTUFBTSxDQUFDO1FBQ3JCQyxVQUFVLEVBQUVILEtBQUssQ0FBQ0csVUFBVSxJQUFJLEdBQUc7UUFDbkNMLElBQUksRUFBRUUsS0FBSyxDQUFDSTtNQUNkLENBQUMsQ0FBQztJQUNKO0VBQ0YsQ0FDRixDQUFDOztFQUVEO0VBQ0F2QixNQUFNLENBQUMrQixJQUFJLENBQ1Q7SUFDRTNCLElBQUksRUFBRyxHQUFFQyx1Q0FBVyx5QkFBd0I7SUFDNUNDLFFBQVEsRUFBRTtNQUNSVyxJQUFJLEVBQUVRLG9CQUFNLENBQUNDLE1BQU0sQ0FBQztRQUNsQmdDLHFCQUFxQixFQUFFakMsb0JBQU0sQ0FBQ2tDLE9BQU8sQ0FBQ2xDLG9CQUFNLENBQUNHLE1BQU0sQ0FBQyxDQUFDO01BQ3ZELENBQUM7SUFDSDtFQUNGLENBQUMsRUFDRCxPQUNFckIsT0FBTyxFQUNQQyxPQUFPLEVBQ1BDLFFBQVEsS0FDd0Q7SUFDaEUsTUFBTUMseUJBQXFELEdBQUdILE9BQU8sQ0FBQ0ksb0JBQW9CLENBQUNDLG1CQUFtQixDQUFDQyxRQUFRLENBQ3JITCxPQUNGLENBQUM7SUFFRCxJQUFJO01BQ0YsTUFBTW9ELFVBQVUsR0FBRyxNQUFNM0Qsa0JBQWtCLENBQUM0RCxlQUFlLENBQ3pEbkQseUJBQXlCLEVBQ3pCRixPQUFPLENBQUNTLElBQUksQ0FBQ3lDLHFCQUNmLENBQUM7TUFDRCxPQUFPakQsUUFBUSxDQUFDTyxFQUFFLENBQUM7UUFDakJDLElBQUksRUFBRTtVQUNKNkMsY0FBYyxFQUFFRjtRQUNsQjtNQUNGLENBQUMsQ0FBQztJQUNKLENBQUMsQ0FBQyxPQUFPekMsS0FBVSxFQUFFO01BQ25CQyxPQUFPLENBQUNELEtBQUssQ0FBQywrQkFBK0IsRUFBRUEsS0FBSyxDQUFDO01BQ3JELE9BQU9WLFFBQVEsQ0FBQ1ksTUFBTSxDQUFDO1FBQ3JCQyxVQUFVLEVBQUVILEtBQUssQ0FBQ0csVUFBVSxJQUFJLEdBQUc7UUFDbkNMLElBQUksRUFBRUUsS0FBSyxDQUFDSTtNQUNkLENBQUMsQ0FBQztJQUNKO0VBQ0YsQ0FDRixDQUFDO0FBQ0gifQ==