"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.registerNoteRoute = registerNoteRoute;
var _configSchema = require("@osd/config-schema");
var _notebooks = require("../../../common/constants/notebooks");
var _notebooks2 = require("../../adaptors/notebooks");
/*
 * Copyright OpenSearch Contributors
 * SPDX-License-Identifier: Apache-2.0
 */

function registerNoteRoute(router) {
  // Fetch all the notebooks available
  router.get({
    path: `${_notebooks.NOTEBOOKS_API_PREFIX}/`,
    validate: {}
  }, async (context, request, response) => {
    const opensearchNotebooksClient = context.observability_plugin.observabilityClient.asScoped(request);
    let notebooksData = [];
    try {
      notebooksData = await _notebooks2.BACKEND.viewNotes(opensearchNotebooksClient, _notebooks.wreckOptions);
      return response.ok({
        body: {
          data: notebooksData
        }
      });
    } catch (error) {
      console.log('Notebook:', error);
      return response.custom({
        statusCode: error.statusCode || 500,
        body: error.message
      });
    }
  });

  // Get all paragraphs of notebooks
  router.get({
    path: `${_notebooks.NOTEBOOKS_API_PREFIX}/note/{noteId}`,
    validate: {
      params: _configSchema.schema.object({
        noteId: _configSchema.schema.string()
      })
    }
  }, async (context, request, response) => {
    const opensearchNotebooksClient = context.observability_plugin.observabilityClient.asScoped(request);
    try {
      const notebookinfo = await _notebooks2.BACKEND.fetchNote(opensearchNotebooksClient, request.params.noteId, _notebooks.wreckOptions);
      return response.ok({
        body: notebookinfo
      });
    } catch (error) {
      return response.custom({
        statusCode: error.statusCode || 500,
        body: error.message
      });
    }
  });

  // Add a Notebook
  router.post({
    path: `${_notebooks.NOTEBOOKS_API_PREFIX}/note`,
    validate: {
      body: _configSchema.schema.object({
        name: _configSchema.schema.string()
      })
    }
  }, async (context, request, response) => {
    const opensearchNotebooksClient = context.observability_plugin.observabilityClient.asScoped(request);
    try {
      const addResponse = await _notebooks2.BACKEND.addNote(opensearchNotebooksClient, request.body, _notebooks.wreckOptions);
      return response.ok({
        body: addResponse.message.objectId
      });
    } catch (error) {
      return response.custom({
        statusCode: error.statusCode || 500,
        body: error.message
      });
    }
  });

  // Rename a notebook
  router.put({
    path: `${_notebooks.NOTEBOOKS_API_PREFIX}/note/rename`,
    validate: {
      body: _configSchema.schema.object({
        name: _configSchema.schema.string(),
        noteId: _configSchema.schema.string()
      })
    }
  }, async (context, request, response) => {
    const opensearchNotebooksClient = context.observability_plugin.observabilityClient.asScoped(request);
    try {
      const renameResponse = await _notebooks2.BACKEND.renameNote(opensearchNotebooksClient, request.body, _notebooks.wreckOptions);
      return response.ok({
        body: renameResponse
      });
    } catch (error) {
      return response.custom({
        statusCode: error.statusCode || 500,
        body: error.message
      });
    }
  });

  // Clone a notebook
  router.post({
    path: `${_notebooks.NOTEBOOKS_API_PREFIX}/note/clone`,
    validate: {
      body: _configSchema.schema.object({
        name: _configSchema.schema.string(),
        noteId: _configSchema.schema.string()
      })
    }
  }, async (context, request, response) => {
    const opensearchNotebooksClient = context.observability_plugin.observabilityClient.asScoped(request);
    try {
      const cloneResponse = await _notebooks2.BACKEND.cloneNote(opensearchNotebooksClient, request.body, _notebooks.wreckOptions);
      return response.ok({
        body: cloneResponse
      });
    } catch (error) {
      return response.custom({
        statusCode: error.statusCode || 500,
        body: error.message
      });
    }
  });

  // Delete notebooks
  router.delete({
    path: `${_notebooks.NOTEBOOKS_API_PREFIX}/note/{noteList}`,
    validate: {
      params: _configSchema.schema.object({
        noteList: _configSchema.schema.string()
      })
    }
  }, async (context, request, response) => {
    const opensearchNotebooksClient = context.observability_plugin.observabilityClient.asScoped(request);
    try {
      const delResponse = await _notebooks2.BACKEND.deleteNote(opensearchNotebooksClient, request.params.noteList, _notebooks.wreckOptions);
      return response.ok({
        body: delResponse
      });
    } catch (error) {
      return response.custom({
        statusCode: error.statusCode || 500,
        body: error.message
      });
    }
  });

  // Add sample notebooks
  router.post({
    path: `${_notebooks.NOTEBOOKS_API_PREFIX}/note/addSampleNotebooks`,
    validate: {
      body: _configSchema.schema.object({
        visIds: _configSchema.schema.arrayOf(_configSchema.schema.string())
      })
    }
  }, async (context, request, response) => {
    const opensearchNotebooksClient = context.observability_plugin.observabilityClient.asScoped(request);
    try {
      const addSampleNotesResponse = await _notebooks2.BACKEND.addSampleNotes(opensearchNotebooksClient, request.body.visIds, _notebooks.wreckOptions);
      return response.ok({
        body: addSampleNotesResponse
      });
    } catch (error) {
      return response.custom({
        statusCode: error.statusCode || 500,
        body: error.message
      });
    }
  });
}
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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