"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.registerParaRoute = registerParaRoute;
var _configSchema = require("@osd/config-schema");
var _notebooks = require("../../../common/constants/notebooks");
var _notebooks2 = require("../../adaptors/notebooks");
/*
 * Copyright OpenSearch Contributors
 * SPDX-License-Identifier: Apache-2.0
 */

function registerParaRoute(router) {
  /* --> Updates the input content in a paragraph
   * --> Runs the paragraph
   * --> Fetches the updated Paragraph (with new input content)
   */
  router.post({
    path: `${_notebooks.NOTEBOOKS_API_PREFIX}/paragraph/update/run/`,
    validate: {
      body: _configSchema.schema.object({
        noteId: _configSchema.schema.string(),
        paragraphId: _configSchema.schema.string(),
        paragraphInput: _configSchema.schema.string(),
        paragraphType: _configSchema.schema.string()
      })
    }
  }, async (context, request, response) => {
    const opensearchNotebooksClient = context.observability_plugin.observabilityClient.asScoped(request);
    try {
      const runResponse = await _notebooks2.BACKEND.updateRunFetchParagraph(context.observability_plugin.observabilityClient, request, _notebooks.wreckOptions);
      return response.ok({
        body: runResponse
      });
    } catch (error) {
      return response.custom({
        statusCode: error.statusCode || 500,
        body: error.message
      });
    }
  });

  /* --> Updates the input content in a paragraph
   * --> Fetches the updated Paragraph (with new input content)
   */
  router.put({
    path: `${_notebooks.NOTEBOOKS_API_PREFIX}/paragraph/`,
    validate: {
      body: _configSchema.schema.object({
        noteId: _configSchema.schema.string(),
        paragraphId: _configSchema.schema.string(),
        paragraphInput: _configSchema.schema.string()
      })
    }
  }, async (context, request, response) => {
    const opensearchNotebooksClient = context.observability_plugin.observabilityClient.asScoped(request);
    try {
      const saveResponse = await _notebooks2.BACKEND.updateFetchParagraph(opensearchNotebooksClient, request.body, _notebooks.wreckOptions);
      return response.ok({
        body: saveResponse
      });
    } catch (error) {
      return response.custom({
        statusCode: error.statusCode || 500,
        body: error.message
      });
    }
  });

  /* --> Adds a new paragraph
   * --> Fetches the added Paragraph
   */
  router.post({
    path: `${_notebooks.NOTEBOOKS_API_PREFIX}/paragraph/`,
    validate: {
      body: _configSchema.schema.object({
        noteId: _configSchema.schema.string(),
        paragraphIndex: _configSchema.schema.number(),
        paragraphInput: _configSchema.schema.string(),
        inputType: _configSchema.schema.string()
      })
    }
  }, async (context, request, response) => {
    const opensearchNotebooksClient = context.observability_plugin.observabilityClient.asScoped(request);
    try {
      const addResponse = await _notebooks2.BACKEND.addFetchNewParagraph(opensearchNotebooksClient, request.body, _notebooks.wreckOptions);
      return response.ok({
        body: addResponse
      });
    } catch (error) {
      return response.custom({
        statusCode: error.statusCode || 500,
        body: error.message
      });
    }
  });

  /* --> Update paragraphs in backend with paragraphs passed in
   * --> Fetches the added Paragraph
   */
  router.post({
    path: `${_notebooks.NOTEBOOKS_API_PREFIX}/set_paragraphs/`,
    validate: {
      body: _configSchema.schema.object({
        noteId: _configSchema.schema.string(),
        paragraphs: _configSchema.schema.arrayOf(_configSchema.schema.object({
          output: _configSchema.schema.maybe(_configSchema.schema.arrayOf(_configSchema.schema.object({}, {
            unknowns: 'allow'
          }))),
          input: _configSchema.schema.object({
            inputText: _configSchema.schema.string(),
            inputType: _configSchema.schema.string()
          }),
          dateCreated: _configSchema.schema.string(),
          dateModified: _configSchema.schema.string(),
          id: _configSchema.schema.string()
        }))
      })
    }
  }, async (context, request, response) => {
    const opensearchNotebooksClient = context.observability_plugin.observabilityClient.asScoped(request);
    try {
      const updateNotebook = {
        paragraphs: request.body.paragraphs,
        dateModified: new Date().toISOString()
      };
      const updateResponse = await _notebooks2.BACKEND.updateNote(opensearchNotebooksClient, request.body.noteId, updateNotebook);
      return response.ok({
        body: updateResponse
      });
    } catch (error) {
      return response.custom({
        statusCode: error.statusCode || 500,
        body: error.message
      });
    }
  });

  /* --> Deletes a paragraph
   * --> Fetches the all other Paragraphs as a list
   * --> Deletes all paragraphs if paragraphId is not provided
   */
  router.delete({
    path: `${_notebooks.NOTEBOOKS_API_PREFIX}/paragraph`,
    validate: {
      query: _configSchema.schema.object({
        noteId: _configSchema.schema.string(),
        paragraphId: _configSchema.schema.maybe(_configSchema.schema.string())
      })
    }
  }, async (context, request, response) => {
    const opensearchNotebooksClient = context.observability_plugin.observabilityClient.asScoped(request);
    const params = {
      noteId: request.query.noteId,
      paragraphId: request.query.paragraphId
    };
    try {
      const deleteResponse = await _notebooks2.BACKEND.deleteFetchParagraphs(opensearchNotebooksClient, params, _notebooks.wreckOptions);
      return response.ok({
        body: deleteResponse
      });
    } catch (error) {
      return response.custom({
        statusCode: error.statusCode || 500,
        body: error.message
      });
    }
  });

  /* --> Clears output for all the paragraphs
   * --> Fetches the all Paragraphs as a list (with cleared outputs)
   */
  router.put({
    path: `${_notebooks.NOTEBOOKS_API_PREFIX}/paragraph/clearall/`,
    validate: {
      body: _configSchema.schema.object({
        noteId: _configSchema.schema.string()
      })
    }
  }, async (context, request, response) => {
    const opensearchNotebooksClient = context.observability_plugin.observabilityClient.asScoped(request);
    try {
      const clearParaResponse = await _notebooks2.BACKEND.clearAllFetchParagraphs(opensearchNotebooksClient, request.body, _notebooks.wreckOptions);
      return response.ok({
        body: clearParaResponse
      });
    } catch (error) {
      return response.custom({
        statusCode: error.statusCode || 500,
        body: error.message
      });
    }
  });
}
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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