"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.BasicAuthRoutes = void 0;
var _configSchema = require("@osd/config-schema");
var _security_cookie = require("../../../session/security_cookie");
var _common = require("../../../../common");
var _tenant_resolver = require("../../../multitenancy/tenant_resolver");
var _encode_uri_query = require("../../../../../../src/plugins/opensearch_dashboards_utils/common/url/encode_uri_query");
/*
 *   Copyright OpenSearch Contributors
 *
 *   Licensed under the Apache License, Version 2.0 (the "License").
 *   You may not use this file except in compliance with the License.
 *   A copy of the License is located at
 *
 *       http://www.apache.org/licenses/LICENSE-2.0
 *
 *   or in the "license" file accompanying this file. This file is distributed
 *   on an "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either
 *   express or implied. See the License for the specific language governing
 *   permissions and limitations under the License.
 */

class BasicAuthRoutes {
  constructor(router, config, sessionStorageFactory, securityClient, coreSetup) {
    this.router = router;
    this.config = config;
    this.sessionStorageFactory = sessionStorageFactory;
    this.securityClient = securityClient;
    this.coreSetup = coreSetup;
  }
  setupRoutes() {
    // bootstrap an empty page so that browser app can render the login page
    // using client side routing.
    this.coreSetup.http.resources.register({
      path: _common.LOGIN_PAGE_URI,
      validate: false,
      options: {
        authRequired: false
      }
    }, async (context, request, response) => {
      this.sessionStorageFactory.asScoped(request).clear();
      const clearOldVersionCookie = (0, _security_cookie.clearOldVersionCookieValue)(this.config);
      return response.renderAnonymousCoreApp({
        headers: {
          'set-cookie': clearOldVersionCookie
        }
      });
    });

    // login using username and password
    this.router.post({
      path: _common.API_AUTH_LOGIN,
      validate: {
        body: _configSchema.schema.object({
          username: _configSchema.schema.string(),
          password: _configSchema.schema.string()
        })
      },
      options: {
        authRequired: false
      }
    }, async (context, request, response) => {
      var _this$config$multiten;
      const forbiddenUsernames = this.config.auth.forbidden_usernames;
      if (forbiddenUsernames.indexOf(request.body.username) > -1) {
        context.security_plugin.logger.error(`Denied login for forbidden username ${request.body.username}`);
        return response.badRequest({
          // Cannot login using forbidden user name.
          body: 'Invalid username or password'
        });
      }
      let user;
      try {
        user = await this.securityClient.authenticate(request, {
          username: request.body.username,
          password: request.body.password
        });
      } catch (error) {
        context.security_plugin.logger.error(`Failed authentication: ${error}`);
        return response.unauthorized({
          headers: {
            'www-authenticate': error.message
          }
        });
      }
      this.sessionStorageFactory.asScoped(request).clear();
      const encodedCredentials = Buffer.from(`${request.body.username}:${request.body.password}`).toString('base64');
      const sessionStorage = {
        username: user.username,
        credentials: {
          authHeaderValue: `Basic ${encodedCredentials}`
        },
        authType: _common.AuthType.BASIC,
        isAnonymousAuth: false,
        expiryTime: Date.now() + this.config.session.ttl
      };
      if (user.multitenancy_enabled) {
        const selectTenant = (0, _tenant_resolver.resolveTenant)({
          request,
          username: user.username,
          roles: user.roles,
          availableTenants: user.tenants,
          config: this.config,
          cookie: sessionStorage,
          multitenancyEnabled: user.multitenancy_enabled,
          privateTenantEnabled: user.private_tenant_enabled,
          defaultTenant: user.default_tenant
        });
        // const selectTenant = user.default_tenant;
        sessionStorage.tenant = selectTenant;
      }
      this.sessionStorageFactory.asScoped(request).set(sessionStorage);
      return response.ok({
        body: {
          username: user.username,
          tenants: user.tenants,
          roles: user.roles,
          backendroles: user.backendRoles,
          selectedTenants: (_this$config$multiten = this.config.multitenancy) !== null && _this$config$multiten !== void 0 && _this$config$multiten.enabled ? sessionStorage.tenant : undefined
        }
      });
    });

    // logout
    this.router.post({
      path: _common.API_AUTH_LOGOUT,
      validate: false,
      options: {
        authRequired: false
      }
    }, async (context, request, response) => {
      this.sessionStorageFactory.asScoped(request).clear();
      return response.ok({
        body: {}
      });
    });

    // anonymous auth
    this.router.get({
      path: _common.ANONYMOUS_AUTH_LOGIN,
      validate: false,
      options: {
        authRequired: false
      }
    }, async (context, request, response) => {
      if (this.config.auth.anonymous_auth_enabled) {
        let user;
        // If the request contains no redirect path, simply redirect to basepath.
        let redirectUrl = this.coreSetup.http.basePath.serverBasePath ? this.coreSetup.http.basePath.serverBasePath : '/';
        const requestQuery = request.url.searchParams;
        const nextUrl = requestQuery === null || requestQuery === void 0 ? void 0 : requestQuery.get('nextUrl');
        if (nextUrl) {
          redirectUrl = nextUrl;
        }
        context.security_plugin.logger.info('The Redirect Path is ' + redirectUrl);
        try {
          user = await this.securityClient.authenticateWithHeaders(request, {});
        } catch (error) {
          context.security_plugin.logger.error(`Failed authentication: ${error}. Redirecting to Login Page`);
          return response.redirected({
            headers: {
              location: `${this.coreSetup.http.basePath.serverBasePath}${_common.LOGIN_PAGE_URI}${nextUrl ? '?nextUrl=' + (0, _encode_uri_query.encodeUriQuery)(redirectUrl) : ''}`
            }
          });
        }
        this.sessionStorageFactory.asScoped(request).clear();
        const sessionStorage = {
          username: user.username,
          authType: _common.AuthType.BASIC,
          isAnonymousAuth: true,
          expiryTime: Date.now() + this.config.session.ttl
        };
        if (user.multitenancy_enabled) {
          const selectTenant = (0, _tenant_resolver.resolveTenant)({
            request,
            username: user.username,
            roles: user.roles,
            availableTenants: user.tenants,
            config: this.config,
            cookie: sessionStorage,
            multitenancyEnabled: user.multitenancy_enabled,
            privateTenantEnabled: user.private_tenant_enabled,
            defaultTenant: user.default_tenant
          });
          sessionStorage.tenant = selectTenant;
        }
        this.sessionStorageFactory.asScoped(request).set(sessionStorage);
        return response.redirected({
          headers: {
            location: `${redirectUrl}`
          }
        });
      } else {
        context.security_plugin.logger.error('Anonymous auth is disabled. Redirecting to Login Page');
        return response.redirected({
          headers: {
            location: `${this.coreSetup.http.basePath.serverBasePath}${_common.LOGIN_PAGE_URI}`
          }
        });
      }
    });
  }
}
exports.BasicAuthRoutes = BasicAuthRoutes;
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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