"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.plugin = exports.TraceStatusCodes = exports.HttpPlugin = void 0;
const core_1 = require("@opencensus/core");
const httpModule = require("http");
const semver = require("semver");
const shimmer = require("shimmer");
const url = require("url");
const uuid = require("uuid");
const express_1 = require("./express");
class HttpPlugin extends core_1.BasePlugin {
    constructor(moduleName) {
        super(moduleName);
    }
    applyPatch() {
        this.logger.debug('applying patch to %s@%s', this.moduleName, this.version);
        shimmer.wrap(this.moduleExports, 'request', this.getPatchOutgoingRequestFunction());
        if (semver.satisfies(this.version, '>=8.0.0')) {
            shimmer.wrap(this.moduleExports, 'get', () => {
                return function getTrace(options, callback) {
                    const req = httpModule.request(options, callback);
                    req.end();
                    return req;
                };
            });
        }
        if (this.moduleExports && this.moduleExports.Server &&
            this.moduleExports.Server.prototype) {
            shimmer.wrap(this.moduleExports.Server.prototype, 'emit', this.getPatchIncomingRequestFunction());
        }
        else {
            this.logger.error('Could not apply patch to %s.emit. Interface is not as expected.', this.moduleName);
        }
        return this.moduleExports;
    }
    applyUnpatch() {
        shimmer.unwrap(this.moduleExports, 'request');
        if (semver.satisfies(this.version, '>=8.0.0')) {
            shimmer.unwrap(this.moduleExports, 'get');
        }
        if (this.moduleExports && this.moduleExports.Server &&
            this.moduleExports.Server.prototype) {
            shimmer.unwrap(this.moduleExports.Server.prototype, 'emit');
        }
    }
    isIgnored(url, request, list) {
        if (!list) {
            return false;
        }
        for (const pattern of list) {
            if (this.isSatisfyPattern(url, request, pattern)) {
                return true;
            }
        }
        return false;
    }
    isSatisfyPattern(url, request, pattern) {
        if (typeof pattern === 'string') {
            return pattern === url;
        }
        else if (pattern instanceof RegExp) {
            return pattern.test(url);
        }
        else if (typeof pattern === 'function') {
            return pattern(url, request);
        }
        else {
            throw new TypeError('Pattern is in unsupported datatype');
        }
    }
    getPatchIncomingRequestFunction() {
        return (original) => {
            const plugin = this;
            return function incomingRequest(event, ...args) {
                if (event !== 'request') {
                    return original.apply(this, arguments);
                }
                const request = args[0];
                const response = args[1];
                const path = url.parse(request.url).pathname;
                plugin.logger.debug('%s plugin incomingRequest', plugin.moduleName);
                if (plugin.isIgnored(path, request, plugin.options.ignoreIncomingPaths)) {
                    return original.apply(this, arguments);
                }
                const propagation = plugin.tracer.propagation;
                const headers = request.headers;
                const getter = {
                    getHeader(name) {
                        return headers[name];
                    }
                };
                const context = propagation ? propagation.extract(getter) : null;
                const traceOptions = {
                    name: path,
                    kind: core_1.SpanKind.SERVER,
                    spanContext: context !== null ? context : undefined
                };
                return plugin.createSpan(traceOptions, rootSpan => {
                    if (!rootSpan)
                        return original.apply(this, arguments);
                    plugin.tracer.wrapEmitter(request);
                    plugin.tracer.wrapEmitter(response);
                    const originalEnd = response.end;
                    response.end = function () {
                        response.end = originalEnd;
                        const returned = response.end.apply(this, arguments);
                        const requestUrl = url.parse(request.url || 'localhost');
                        const host = headers.host || 'localhost';
                        const userAgent = (headers['user-agent'] || headers['User-Agent']);
                        rootSpan.addAttribute(HttpPlugin.ATTRIBUTE_HTTP_HOST, host.replace(/^(.*)(\:[0-9]{1,5})/, '$1'));
                        rootSpan.addAttribute(HttpPlugin.ATTRIBUTE_HTTP_METHOD, request.method || 'GET');
                        rootSpan.addAttribute(HttpPlugin.ATTRIBUTE_HTTP_PATH, `${requestUrl.pathname}`);
                        let route = `${requestUrl.path}`;
                        const middlewareStack = request[express_1.kMiddlewareStack];
                        if (middlewareStack) {
                            route = middlewareStack
                                .filter(path => path !== '/')
                                .map(path => {
                                return path[0] === '/' ? path : '/' + path;
                            }).join('');
                        }
                        rootSpan.addAttribute(HttpPlugin.ATTRIBUTE_HTTP_ROUTE, route);
                        rootSpan.addAttribute(HttpPlugin.ATTRIBUTE_HTTP_USER_AGENT, userAgent);
                        rootSpan.addAttribute(HttpPlugin.ATTRIBUTE_HTTP_STATUS_CODE, response.statusCode.toString());
                        rootSpan.setStatus(HttpPlugin.convertTraceStatus(response.statusCode));
                        rootSpan.addMessageEvent(core_1.MessageEventType.RECEIVED, uuid.v4().split('-').join(''));
                        rootSpan.end();
                        return returned;
                    };
                    return original.apply(this, arguments);
                });
            };
        };
    }
    getPatchOutgoingRequestFunction() {
        return (original) => {
            const plugin = this;
            const kind = plugin.moduleName === 'https' ? 'HTTPS' : 'HTTP';
            return function outgoingRequest(options, callback) {
                if (!options) {
                    return original.apply(this, arguments);
                }
                let pathname = '';
                let method = 'GET';
                let origin = '';
                if (typeof (options) === 'string') {
                    const parsedUrl = url.parse(options);
                    options = parsedUrl;
                    pathname = parsedUrl.pathname || '/';
                    origin = `${parsedUrl.protocol || 'http:'}//${parsedUrl.host}`;
                }
                else {
                    if (options.headers &&
                        options.headers['x-opencensus-outgoing-request']) {
                        plugin.logger.debug('header with "x-opencensus-outgoing-request" - do not trace');
                        return original.apply(this, arguments);
                    }
                    try {
                        pathname = options.pathname || '';
                        if (pathname.length === 0 && typeof options.path === 'string') {
                            pathname = url.parse(options.path).pathname || '';
                        }
                        method = options.method || 'GET';
                        origin = `${options.protocol || 'http:'}//${options.host}`;
                    }
                    catch (e) {
                        return original.apply(this, arguments);
                    }
                }
                const request = original.apply(this, arguments);
                if (plugin.isIgnored(origin + pathname, request, plugin.options.ignoreOutgoingUrls)) {
                    return request;
                }
                plugin.tracer.wrapEmitter(request);
                plugin.logger.debug('%s plugin outgoingRequest', plugin.moduleName);
                const traceOptions = {
                    name: `${kind.toLowerCase()}-${(method || 'GET').toLowerCase()}`,
                    kind: core_1.SpanKind.CLIENT
                };
                if (!plugin.tracer.currentRootSpan) {
                    plugin.logger.debug('outgoingRequest starting a root span');
                    return plugin.tracer.startRootSpan(traceOptions, plugin.getMakeRequestTraceFunction(request, options, plugin));
                }
                else {
                    plugin.logger.debug('outgoingRequest starting a child span');
                    const span = plugin.tracer.startChildSpan(traceOptions.name, traceOptions.kind);
                    return (plugin.getMakeRequestTraceFunction(request, options, plugin))(span);
                }
            };
        };
    }
    getMakeRequestTraceFunction(request, options, plugin) {
        return (span) => {
            plugin.logger.debug('makeRequestTrace');
            if (!span) {
                plugin.logger.debug('makeRequestTrace span is null');
                return request;
            }
            const setter = {
                setHeader(name, value) {
                    if (plugin.hasExpectHeader(options) && options.headers) {
                        if (options.__cloned !== true) {
                            options = Object.assign({}, options);
                            options.headers = Object.assign({}, options.headers);
                            options.__cloned = true;
                        }
                        options.headers[name] = value;
                    }
                    else {
                        request.setHeader(name, value);
                    }
                }
            };
            const propagation = plugin.tracer.propagation;
            if (propagation) {
                propagation.inject(setter, span.spanContext);
            }
            request.on('response', (response) => {
                plugin.tracer.wrapEmitter(response);
                plugin.logger.debug('outgoingRequest on response()');
                response.on('end', () => {
                    plugin.logger.debug('outgoingRequest on end()');
                    const method = response.method ? response.method : 'GET';
                    const headers = options.headers;
                    const userAgent = headers ? (headers['user-agent'] || headers['User-Agent']) : null;
                    if (options.host || options.hostname) {
                        const value = options.host || options.hostname;
                        span.addAttribute(HttpPlugin.ATTRIBUTE_HTTP_HOST, `${value}`);
                    }
                    span.addAttribute(HttpPlugin.ATTRIBUTE_HTTP_METHOD, method);
                    span.addAttribute(HttpPlugin.ATTRIBUTE_HTTP_PATH, `${options.path}`);
                    span.addAttribute(HttpPlugin.ATTRIBUTE_HTTP_ROUTE, `${options.path}`);
                    if (userAgent) {
                        span.addAttribute(HttpPlugin.ATTRIBUTE_HTTP_USER_AGENT, userAgent.toString());
                    }
                    span.addAttribute(HttpPlugin.ATTRIBUTE_HTTP_STATUS_CODE, `${response.statusCode}`);
                    span.setStatus(HttpPlugin.convertTraceStatus(response.statusCode || 0));
                    span.addMessageEvent(core_1.MessageEventType.SENT, uuid.v4().split('-').join(''));
                    span.end();
                });
                response.on('error', error => {
                    span.addAttribute(HttpPlugin.ATTRIBUTE_HTTP_ERROR_NAME, error.name);
                    span.addAttribute(HttpPlugin.ATTRIBUTE_HTTP_ERROR_MESSAGE, error.message);
                    span.setStatus(core_1.CanonicalCode.UNKNOWN);
                    span.end();
                });
            });
            request.on('error', error => {
                span.addAttribute(HttpPlugin.ATTRIBUTE_HTTP_ERROR_NAME, error.name);
                span.addAttribute(HttpPlugin.ATTRIBUTE_HTTP_ERROR_MESSAGE, error.message);
                span.setStatus(core_1.CanonicalCode.UNKNOWN);
                span.end();
            });
            plugin.logger.debug('makeRequestTrace return request');
            return request;
        };
    }
    createSpan(options, fn) {
        const forceChildspan = this.options.createSpanWithNet === true;
        if (forceChildspan) {
            const span = this.tracer.startChildSpan(options.name, options.kind);
            return fn(span);
        }
        else {
            return this.tracer.startRootSpan(options, fn);
        }
    }
    static convertTraceStatus(statusCode) {
        if (statusCode < 200 || statusCode > 504) {
            return TraceStatusCodes.UNKNOWN;
        }
        else if (statusCode >= 200 && statusCode < 400) {
            return TraceStatusCodes.OK;
        }
        else {
            switch (statusCode) {
                case (400):
                    return TraceStatusCodes.INVALID_ARGUMENT;
                case (504):
                    return TraceStatusCodes.DEADLINE_EXCEEDED;
                case (404):
                    return TraceStatusCodes.NOT_FOUND;
                case (403):
                    return TraceStatusCodes.PERMISSION_DENIED;
                case (401):
                    return TraceStatusCodes.UNAUTHENTICATED;
                case (429):
                    return TraceStatusCodes.RESOURCE_EXHAUSTED;
                case (501):
                    return TraceStatusCodes.UNIMPLEMENTED;
                case (503):
                    return TraceStatusCodes.UNAVAILABLE;
                default:
                    return TraceStatusCodes.UNKNOWN;
            }
        }
    }
    hasExpectHeader(options) {
        return !!(options.headers &&
            options.headers.Expect);
    }
}
exports.HttpPlugin = HttpPlugin;
HttpPlugin.ATTRIBUTE_HTTP_HOST = 'http.host';
HttpPlugin.ATTRIBUTE_HTTP_METHOD = 'http.method';
HttpPlugin.ATTRIBUTE_HTTP_PATH = 'http.path';
HttpPlugin.ATTRIBUTE_HTTP_ROUTE = 'http.route';
HttpPlugin.ATTRIBUTE_HTTP_USER_AGENT = 'http.user_agent';
HttpPlugin.ATTRIBUTE_HTTP_STATUS_CODE = 'http.status_code';
HttpPlugin.ATTRIBUTE_HTTP_ERROR_NAME = 'http.error_name';
HttpPlugin.ATTRIBUTE_HTTP_ERROR_MESSAGE = 'http.error_message';
var TraceStatusCodes;
(function (TraceStatusCodes) {
    TraceStatusCodes[TraceStatusCodes["UNKNOWN"] = 2] = "UNKNOWN";
    TraceStatusCodes[TraceStatusCodes["OK"] = 0] = "OK";
    TraceStatusCodes[TraceStatusCodes["INVALID_ARGUMENT"] = 3] = "INVALID_ARGUMENT";
    TraceStatusCodes[TraceStatusCodes["DEADLINE_EXCEEDED"] = 4] = "DEADLINE_EXCEEDED";
    TraceStatusCodes[TraceStatusCodes["NOT_FOUND"] = 5] = "NOT_FOUND";
    TraceStatusCodes[TraceStatusCodes["PERMISSION_DENIED"] = 7] = "PERMISSION_DENIED";
    TraceStatusCodes[TraceStatusCodes["UNAUTHENTICATED"] = 16] = "UNAUTHENTICATED";
    TraceStatusCodes[TraceStatusCodes["RESOURCE_EXHAUSTED"] = 8] = "RESOURCE_EXHAUSTED";
    TraceStatusCodes[TraceStatusCodes["UNIMPLEMENTED"] = 12] = "UNIMPLEMENTED";
    TraceStatusCodes[TraceStatusCodes["UNAVAILABLE"] = 14] = "UNAVAILABLE";
})(TraceStatusCodes || (exports.TraceStatusCodes = TraceStatusCodes = {}));
exports.plugin = new HttpPlugin('http');
//# sourceMappingURL=data:application/json;base64,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