"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.plugin = exports.MongoDBPlugin = void 0;
const core_1 = require("@opencensus/core");
const shimmer = require("shimmer");
class MongoDBPlugin extends core_1.BasePlugin {
    constructor(moduleName) {
        super(moduleName);
        this.internalFileList = {
            '1 - 3': {
                'ConnectionPool': 'lib/connection/pool'
            }
        };
    }
    applyPatch() {
        this.logger.debug('Patched MongoDB');
        if (this.moduleExports.Server) {
            this.logger.debug('patching mongodb-core.Server.prototype functions: insert, remove, command, update');
            shimmer.wrap(this.moduleExports.Server.prototype, 'insert', this.getPatchCommand('mongodb-insert'));
            shimmer.wrap(this.moduleExports.Server.prototype, 'remove', this.getPatchCommand('mongodb-remove'));
            shimmer.wrap(this.moduleExports.Server.prototype, 'command', this.getPatchCommand('mongodb-command'));
            shimmer.wrap(this.moduleExports.Server.prototype, 'update', this.getPatchCommand('mongodb-update'));
        }
        if (this.moduleExports.Cursor) {
            this.logger.debug('patching mongodb-core.Cursor.prototype.next');
            shimmer.wrap(this.moduleExports.Cursor.prototype, 'next', this.getPatchCursor());
        }
        if (this.internalFilesExports.ConnectionPool) {
            this.logger.debug('patching mongodb-core/lib/connection/pool');
            shimmer.wrap(this.internalFilesExports.ConnectionPool.prototype, 'once', this.getPatchEventEmitter());
        }
        return this.moduleExports;
    }
    applyUnpatch() {
        shimmer.unwrap(this.moduleExports.Server.prototype, 'insert');
        shimmer.unwrap(this.moduleExports.Server.prototype, 'remove');
        shimmer.unwrap(this.moduleExports.Server.prototype, 'command');
        shimmer.unwrap(this.moduleExports.Server.prototype, 'update');
        shimmer.unwrap(this.moduleExports.Cursor.prototype, 'next');
        if (this.internalFilesExports.ConnectionPool) {
            shimmer.unwrap(this.internalFilesExports.ConnectionPool.prototype, 'once');
        }
    }
    getPatchCommand(label) {
        const plugin = this;
        return (original) => {
            return function (ns, command, options, callback) {
                const resultHandler = typeof options === 'function' ? options : callback;
                if (plugin.tracer.currentRootSpan && typeof resultHandler === 'function') {
                    let type;
                    if (command.createIndexes) {
                        type = 'createIndexes';
                    }
                    else if (command.findandmodify) {
                        type = 'findAndModify';
                    }
                    else if (command.ismaster) {
                        type = 'isMaster';
                    }
                    else if (command.count) {
                        type = 'count';
                    }
                    else {
                        type = 'command';
                    }
                    const span = plugin.tracer.startChildSpan(label, core_1.SpanKind.CLIENT);
                    if (span === null)
                        return original.apply(this, arguments);
                    span.addAttribute('database', ns);
                    span.addAttribute('type', type);
                    if (plugin.options.detailedCommands === true) {
                        span.addAttribute('command', JSON.stringify(command));
                    }
                    if (typeof options === 'function') {
                        return original.call(this, ns, command, plugin.patchEnd(span, options));
                    }
                    else {
                        return original.call(this, ns, command, options, plugin.patchEnd(span, callback));
                    }
                }
                return original.apply(this, arguments);
            };
        };
    }
    getPatchCursor() {
        const plugin = this;
        return (original) => {
            return function (...args) {
                let resultHandler = args[0];
                if (plugin.tracer.currentRootSpan && typeof resultHandler === 'function') {
                    const span = plugin.tracer.startChildSpan('mongodb-find', core_1.SpanKind.CLIENT);
                    if (span === null)
                        return original.apply(this, arguments);
                    resultHandler = plugin.patchEnd(span, resultHandler);
                    span.addAttribute('database', this.ns);
                    if (plugin.options.detailedCommands === true && typeof this.cmd.query === 'object') {
                        span.addAttribute('command', JSON.stringify(this.cmd.query));
                    }
                }
                return original.call(this, resultHandler);
            };
        };
    }
    getPatchEventEmitter() {
        const plugin = this;
        return (original) => {
            return function (event, cb) {
                return original.call(this, event, plugin.tracer.wrap(cb));
            };
        };
    }
    patchEnd(span, resultHandler) {
        const plugin = this;
        const patchedEnd = function (err, res) {
            if (plugin.options.detailedCommands === true && err instanceof Error) {
                span.addAttribute('error', err.message);
            }
            if (span.ended === false) {
                span.end();
            }
            return resultHandler.apply(this, arguments);
        };
        return this.tracer.wrap(patchedEnd);
    }
}
exports.MongoDBPlugin = MongoDBPlugin;
const plugin = new MongoDBPlugin('mongodb-core');
exports.plugin = plugin;
//# sourceMappingURL=data:application/json;base64,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