"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.plugin = exports.PGPlugin = void 0;
const EventEmitter = require("events");
const core_1 = require("@opencensus/core");
const shimmer = require("shimmer");
class PGPlugin extends core_1.BasePlugin {
    constructor(moduleName) {
        super(moduleName);
        this.internalFileList = {
            '6 - 7': {
                'client': 'lib/client'
            }
        };
    }
    applyPatch() {
        this.logger.debug('Patched PG');
        if (this.internalFilesExports.client) {
            this.logger.debug('patching pq.client.prototype.query');
            shimmer.wrap(this.internalFilesExports.client.prototype, 'query', this.getPatchCreateQuery());
        }
        return this.moduleExports;
    }
    applyUnpatch() {
        shimmer.unwrap(this.internalFilesExports.client.prototype, 'query');
    }
    getPatchCreateQuery() {
        const plugin = this;
        return (original) => {
            return function (...args) {
                const span = plugin.tracer.startChildSpan('pg-query', core_1.SpanKind.CLIENT);
                if (span === null)
                    return original.apply(this, arguments);
                let pgQuery;
                if (arguments.length >= 1) {
                    const args = Array.prototype.slice.call(arguments, 0);
                    plugin.populateLabelsFromInputs(span, args);
                    const callback = args[args.length - 1];
                    if (typeof callback === 'function') {
                        args[args.length - 1] = plugin.patchCallback(callback, span);
                    }
                    else if (typeof args[0] === 'object') {
                        plugin.patchSubmittable(args[0], span);
                    }
                    pgQuery = original.apply(this, args);
                }
                else {
                    pgQuery = original.apply(this, arguments);
                }
                if (pgQuery) {
                    if (pgQuery instanceof EventEmitter) {
                        plugin.tracer.wrapEmitter(pgQuery);
                    }
                    else if (typeof pgQuery.then === 'function') {
                        plugin.patchPromise(pgQuery, span);
                    }
                }
                return pgQuery;
            };
        };
    }
    patchCallback(callback, span) {
        const plugin = this;
        return plugin.tracer.wrap((err, res) => {
            plugin.populateLabelsFromOutputs(span, err, res);
            span.end();
            callback(err, res);
        });
    }
    patchSubmittable(pgQuery, span) {
        const plugin = this;
        let spanEnded = false;
        if (pgQuery.handleError) {
            shimmer.wrap(pgQuery, 'handleError', (origCallback) => {
                return this.tracer.wrap(function (...args) {
                    if (!spanEnded) {
                        const err = args[0];
                        plugin.populateLabelsFromOutputs(span, err);
                        span.end();
                        spanEnded = true;
                    }
                    if (origCallback) {
                        origCallback.apply(this, args);
                    }
                });
            });
        }
        if (pgQuery.handleReadyForQuery) {
            shimmer.wrap(pgQuery, 'handleReadyForQuery', (origCallback) => {
                return this.tracer.wrap(function (...args) {
                    if (!spanEnded) {
                        plugin.populateLabelsFromOutputs(span, null, this._result);
                        span.end();
                        spanEnded = true;
                    }
                    if (origCallback) {
                        origCallback.apply(this, args);
                    }
                });
            });
        }
        return pgQuery;
    }
    patchPromise(promise, span) {
        return promise = promise.then((res) => {
            plugin.populateLabelsFromOutputs(span, null, res);
            span.end();
            return res;
        }, (err) => {
            plugin.populateLabelsFromOutputs(span, err);
            span.end();
            throw err;
        });
    }
    populateLabelsFromInputs(span, args) {
        const queryObj = args[0];
        if (typeof queryObj === 'object') {
            if (queryObj.text) {
                span.addAttribute('query', queryObj.text);
            }
            if (plugin.options.detailedCommands === true && queryObj.values) {
                span.addAttribute('values', queryObj.values);
            }
        }
        else if (typeof queryObj === 'string') {
            span.addAttribute('query', queryObj);
            if (plugin.options.detailedCommands === true && args.length >= 2 && typeof args[1] !== 'function') {
                span.addAttribute('values', args[1]);
            }
        }
    }
    populateLabelsFromOutputs(span, err, res) {
        if (plugin.options.detailedCommands !== true)
            return;
        if (err) {
            span.addAttribute('error', err.toString());
        }
        if (res) {
            span.addAttribute('row_count', res.rowCount);
            span.addAttribute('oid', res.oid);
            span.addAttribute('rows', res.rows);
            span.addAttribute('fields', res.fields);
        }
    }
}
exports.PGPlugin = PGPlugin;
const plugin = new PGPlugin('pg');
exports.plugin = plugin;
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoicGcuanMiLCJzb3VyY2VSb290IjoiIiwic291cmNlcyI6WyIuLi8uLi8uLi8uLi9zcmMvY2Vuc3VzL3BsdWdpbnMvcGcudHMiXSwibmFtZXMiOltdLCJtYXBwaW5ncyI6Ijs7O0FBZUEsdUNBQXNDO0FBRXRDLDJDQUE2RDtBQUM3RCxtQ0FBa0M7QUFVbEMsTUFBYSxRQUFTLFNBQVEsaUJBQVU7SUFVdEMsWUFBYSxVQUFrQjtRQUM3QixLQUFLLENBQUMsVUFBVSxDQUFDLENBQUE7UUFSQSxxQkFBZ0IsR0FBRztZQUNwQyxPQUFPLEVBQUU7Z0JBQ1AsUUFBUSxFQUFFLFlBQVk7YUFDdkI7U0FDRixDQUFBO0lBS0QsQ0FBQztJQUtTLFVBQVU7UUFDbEIsSUFBSSxDQUFDLE1BQU0sQ0FBQyxLQUFLLENBQUMsWUFBWSxDQUFDLENBQUE7UUFFL0IsSUFBSSxJQUFJLENBQUMsb0JBQW9CLENBQUMsTUFBTSxFQUFFLENBQUM7WUFDckMsSUFBSSxDQUFDLE1BQU0sQ0FBQyxLQUFLLENBQUMsb0NBQW9DLENBQUMsQ0FBQTtZQUN2RCxPQUFPLENBQUMsSUFBSSxDQUFDLElBQUksQ0FBQyxvQkFBb0IsQ0FBQyxNQUFNLENBQUMsU0FBUyxFQUFFLE9BQU8sRUFBRSxJQUFJLENBQUMsbUJBQW1CLEVBQUUsQ0FBQyxDQUFBO1FBQy9GLENBQUM7UUFFRCxPQUFPLElBQUksQ0FBQyxhQUFhLENBQUE7SUFDM0IsQ0FBQztJQUdELFlBQVk7UUFDVixPQUFPLENBQUMsTUFBTSxDQUFDLElBQUksQ0FBQyxvQkFBb0IsQ0FBQyxNQUFNLENBQUMsU0FBUyxFQUFFLE9BQU8sQ0FBQyxDQUFBO0lBQ3JFLENBQUM7SUFFTyxtQkFBbUI7UUFDekIsTUFBTSxNQUFNLEdBQUcsSUFBSSxDQUFBO1FBQ25CLE9BQU8sQ0FBQyxRQUFrQixFQUFFLEVBQUU7WUFDNUIsT0FBTyxVQUFVLEdBQUcsSUFBVztnQkFDN0IsTUFBTSxJQUFJLEdBQUcsTUFBTSxDQUFDLE1BQU0sQ0FBQyxjQUFjLENBQUMsVUFBVSxFQUFFLGVBQVEsQ0FBQyxNQUFNLENBQUMsQ0FBQTtnQkFDdEUsSUFBSSxJQUFJLEtBQUssSUFBSTtvQkFBRSxPQUFPLFFBQVEsQ0FBQyxLQUFLLENBQUMsSUFBSSxFQUFFLFNBQVMsQ0FBQyxDQUFBO2dCQUV6RCxJQUFJLE9BQU8sQ0FBQTtnQkFFWCxJQUFJLFNBQVMsQ0FBQyxNQUFNLElBQUksQ0FBQyxFQUFFLENBQUM7b0JBQzFCLE1BQU0sSUFBSSxHQUFHLEtBQUssQ0FBQyxTQUFTLENBQUMsS0FBSyxDQUFDLElBQUksQ0FBQyxTQUFTLEVBQUUsQ0FBQyxDQUFDLENBQUE7b0JBR3JELE1BQU0sQ0FBQyx3QkFBd0IsQ0FBQyxJQUFJLEVBQUUsSUFBSSxDQUFDLENBQUE7b0JBSTNDLE1BQU0sUUFBUSxHQUFHLElBQUksQ0FBQyxJQUFJLENBQUMsTUFBTSxHQUFHLENBQUMsQ0FBQyxDQUFBO29CQUN0QyxJQUFJLE9BQU8sUUFBUSxLQUFLLFVBQVUsRUFBRSxDQUFDO3dCQUNuQyxJQUFJLENBQUMsSUFBSSxDQUFDLE1BQU0sR0FBRyxDQUFDLENBQUMsR0FBRyxNQUFNLENBQUMsYUFBYSxDQUFDLFFBQVEsRUFBRSxJQUFJLENBQUMsQ0FBQTtvQkFDOUQsQ0FBQzt5QkFBTSxJQUFJLE9BQU8sSUFBSSxDQUFDLENBQUMsQ0FBQyxLQUFLLFFBQVEsRUFBRSxDQUFDO3dCQUN2QyxNQUFNLENBQUMsZ0JBQWdCLENBQUMsSUFBSSxDQUFDLENBQUMsQ0FBQyxFQUFFLElBQUksQ0FBQyxDQUFBO29CQUN4QyxDQUFDO29CQUNELE9BQU8sR0FBRyxRQUFRLENBQUMsS0FBSyxDQUFDLElBQUksRUFBRSxJQUFJLENBQUMsQ0FBQTtnQkFDdEMsQ0FBQztxQkFBTSxDQUFDO29CQUNOLE9BQU8sR0FBRyxRQUFRLENBQUMsS0FBSyxDQUFDLElBQUksRUFBRSxTQUFTLENBQUMsQ0FBQTtnQkFDM0MsQ0FBQztnQkFFRCxJQUFJLE9BQU8sRUFBRSxDQUFDO29CQUNaLElBQUksT0FBTyxZQUFZLFlBQVksRUFBRSxDQUFDO3dCQUNwQyxNQUFNLENBQUMsTUFBTSxDQUFDLFdBQVcsQ0FBQyxPQUFPLENBQUMsQ0FBQTtvQkFDcEMsQ0FBQzt5QkFBTSxJQUFJLE9BQU8sT0FBTyxDQUFDLElBQUksS0FBSyxVQUFVLEVBQUUsQ0FBQzt3QkFDOUMsTUFBTSxDQUFDLFlBQVksQ0FBQyxPQUFPLEVBQUUsSUFBSSxDQUFDLENBQUE7b0JBQ3BDLENBQUM7Z0JBQ0gsQ0FBQztnQkFDRCxPQUFPLE9BQU8sQ0FBQTtZQUNoQixDQUFDLENBQUE7UUFDSCxDQUFDLENBQUE7SUFDSCxDQUFDO0lBRU8sYUFBYSxDQUFFLFFBQVEsRUFBRSxJQUFJO1FBQ25DLE1BQU0sTUFBTSxHQUFHLElBQUksQ0FBQTtRQUNuQixPQUFPLE1BQU0sQ0FBQyxNQUFNLENBQUMsSUFBSSxDQUFDLENBQUMsR0FBRyxFQUFFLEdBQUcsRUFBRSxFQUFFO1lBQ3JDLE1BQU0sQ0FBQyx5QkFBeUIsQ0FBQyxJQUFJLEVBQUUsR0FBRyxFQUFFLEdBQUcsQ0FBQyxDQUFBO1lBQ2hELElBQUksQ0FBQyxHQUFHLEVBQUUsQ0FBQTtZQUNWLFFBQVEsQ0FBQyxHQUFHLEVBQUUsR0FBRyxDQUFDLENBQUE7UUFDcEIsQ0FBQyxDQUFDLENBQUE7SUFDSixDQUFDO0lBRU8sZ0JBQWdCLENBQUUsT0FBTyxFQUFFLElBQUk7UUFDckMsTUFBTSxNQUFNLEdBQUcsSUFBSSxDQUFBO1FBQ25CLElBQUksU0FBUyxHQUFHLEtBQUssQ0FBQTtRQUNyQixJQUFJLE9BQU8sQ0FBQyxXQUFXLEVBQUUsQ0FBQztZQUN4QixPQUFPLENBQUMsSUFBSSxDQUFDLE9BQU8sRUFBRSxhQUFhLEVBQUUsQ0FBQyxZQUFZLEVBQUUsRUFBRTtnQkFHcEQsT0FBTyxJQUFJLENBQUMsTUFBTSxDQUFDLElBQUksQ0FBQyxVQUFnQixHQUFHLElBQVc7b0JBRXBELElBQUksQ0FBQyxTQUFTLEVBQUUsQ0FBQzt3QkFDZixNQUFNLEdBQUcsR0FBVSxJQUFJLENBQUMsQ0FBQyxDQUFDLENBQUE7d0JBQzFCLE1BQU0sQ0FBQyx5QkFBeUIsQ0FBQyxJQUFJLEVBQUUsR0FBRyxDQUFDLENBQUE7d0JBQzNDLElBQUksQ0FBQyxHQUFHLEVBQUUsQ0FBQTt3QkFDVixTQUFTLEdBQUcsSUFBSSxDQUFBO29CQUNsQixDQUFDO29CQUNELElBQUksWUFBWSxFQUFFLENBQUM7d0JBQ2pCLFlBQVksQ0FBQyxLQUFLLENBQUMsSUFBSSxFQUFFLElBQUksQ0FBQyxDQUFBO29CQUNoQyxDQUFDO2dCQUNILENBQUMsQ0FBQyxDQUFBO1lBQ0osQ0FBQyxDQUFDLENBQUE7UUFDSixDQUFDO1FBQ0QsSUFBSSxPQUFPLENBQUMsbUJBQW1CLEVBQUUsQ0FBQztZQUNoQyxPQUFPLENBQUMsSUFBSSxDQUFDLE9BQU8sRUFBRSxxQkFBcUIsRUFBRSxDQUFDLFlBQVksRUFBRSxFQUFFO2dCQUc1RCxPQUFPLElBQUksQ0FBQyxNQUFNLENBQUMsSUFBSSxDQUFDLFVBQWdCLEdBQUcsSUFBVztvQkFFcEQsSUFBSSxDQUFDLFNBQVMsRUFBRSxDQUFDO3dCQUNmLE1BQU0sQ0FBQyx5QkFBeUIsQ0FBQyxJQUFJLEVBQUUsSUFBSSxFQUFFLElBQUksQ0FBQyxPQUFPLENBQUMsQ0FBQTt3QkFDMUQsSUFBSSxDQUFDLEdBQUcsRUFBRSxDQUFBO3dCQUNWLFNBQVMsR0FBRyxJQUFJLENBQUE7b0JBQ2xCLENBQUM7b0JBQ0QsSUFBSSxZQUFZLEVBQUUsQ0FBQzt3QkFDakIsWUFBWSxDQUFDLEtBQUssQ0FBQyxJQUFJLEVBQUUsSUFBSSxDQUFDLENBQUE7b0JBQ2hDLENBQUM7Z0JBQ0gsQ0FBQyxDQUFDLENBQUE7WUFDSixDQUFDLENBQUMsQ0FBQTtRQUNKLENBQUM7UUFDRCxPQUFPLE9BQU8sQ0FBQTtJQUNoQixDQUFDO0lBRU8sWUFBWSxDQUFFLE9BQU8sRUFBRSxJQUFJO1FBQ2pDLE9BQU8sT0FBTyxHQUFHLE9BQU8sQ0FBQyxJQUFJLENBQUMsQ0FBQyxHQUFHLEVBQUUsRUFBRTtZQUNwQyxNQUFNLENBQUMseUJBQXlCLENBQUMsSUFBSSxFQUFFLElBQUksRUFBRSxHQUFHLENBQUMsQ0FBQTtZQUNqRCxJQUFJLENBQUMsR0FBRyxFQUFFLENBQUE7WUFDVixPQUFPLEdBQUcsQ0FBQTtRQUNaLENBQUMsRUFDRCxDQUFDLEdBQUcsRUFBRSxFQUFFO1lBQ04sTUFBTSxDQUFDLHlCQUF5QixDQUFDLElBQUksRUFBRSxHQUFHLENBQUMsQ0FBQTtZQUMzQyxJQUFJLENBQUMsR0FBRyxFQUFFLENBQUE7WUFDVixNQUFNLEdBQUcsQ0FBQTtRQUNYLENBQUMsQ0FBQyxDQUFBO0lBQ0osQ0FBQztJQUVPLHdCQUF3QixDQUFFLElBQVUsRUFBRSxJQUFTO1FBQ3JELE1BQU0sUUFBUSxHQUFHLElBQUksQ0FBQyxDQUFDLENBQUMsQ0FBQTtRQUN4QixJQUFJLE9BQU8sUUFBUSxLQUFLLFFBQVEsRUFBRSxDQUFDO1lBQ2pDLElBQUksUUFBUSxDQUFDLElBQUksRUFBRSxDQUFDO2dCQUNsQixJQUFJLENBQUMsWUFBWSxDQUFDLE9BQU8sRUFBRSxRQUFRLENBQUMsSUFBSSxDQUFDLENBQUE7WUFDM0MsQ0FBQztZQUNELElBQUksTUFBTSxDQUFDLE9BQU8sQ0FBQyxnQkFBZ0IsS0FBSyxJQUFJLElBQUksUUFBUSxDQUFDLE1BQU0sRUFBRSxDQUFDO2dCQUNoRSxJQUFJLENBQUMsWUFBWSxDQUFDLFFBQVEsRUFBRSxRQUFRLENBQUMsTUFBTSxDQUFDLENBQUE7WUFDOUMsQ0FBQztRQUNILENBQUM7YUFBTSxJQUFJLE9BQU8sUUFBUSxLQUFLLFFBQVEsRUFBRSxDQUFDO1lBQ3hDLElBQUksQ0FBQyxZQUFZLENBQUMsT0FBTyxFQUFFLFFBQVEsQ0FBQyxDQUFBO1lBQ3BDLElBQUksTUFBTSxDQUFDLE9BQU8sQ0FBQyxnQkFBZ0IsS0FBSyxJQUFJLElBQUksSUFBSSxDQUFDLE1BQU0sSUFBSSxDQUFDLElBQUksT0FBTyxJQUFJLENBQUMsQ0FBQyxDQUFDLEtBQUssVUFBVSxFQUFFLENBQUM7Z0JBQ2xHLElBQUksQ0FBQyxZQUFZLENBQUMsUUFBUSxFQUFFLElBQUksQ0FBQyxDQUFDLENBQUMsQ0FBQyxDQUFBO1lBQ3RDLENBQUM7UUFDSCxDQUFDO0lBQ0gsQ0FBQztJQUVPLHlCQUF5QixDQUFFLElBQVUsRUFBRSxHQUFpQixFQUFFLEdBQVM7UUFDekUsSUFBSSxNQUFNLENBQUMsT0FBTyxDQUFDLGdCQUFnQixLQUFLLElBQUk7WUFBRSxPQUFNO1FBRXBELElBQUksR0FBRyxFQUFFLENBQUM7WUFDUixJQUFJLENBQUMsWUFBWSxDQUFDLE9BQU8sRUFBRSxHQUFHLENBQUMsUUFBUSxFQUFFLENBQUMsQ0FBQTtRQUM1QyxDQUFDO1FBQ0QsSUFBSSxHQUFHLEVBQUUsQ0FBQztZQUNSLElBQUksQ0FBQyxZQUFZLENBQUMsV0FBVyxFQUFFLEdBQUcsQ0FBQyxRQUFRLENBQUMsQ0FBQTtZQUM1QyxJQUFJLENBQUMsWUFBWSxDQUFDLEtBQUssRUFBRSxHQUFHLENBQUMsR0FBRyxDQUFDLENBQUE7WUFDakMsSUFBSSxDQUFDLFlBQVksQ0FBQyxNQUFNLEVBQUUsR0FBRyxDQUFDLElBQUksQ0FBQyxDQUFBO1lBQ25DLElBQUksQ0FBQyxZQUFZLENBQUMsUUFBUSxFQUFFLEdBQUcsQ0FBQyxNQUFNLENBQUMsQ0FBQTtRQUN6QyxDQUFDO0lBQ0gsQ0FBQztDQUNGO0FBdEtELDRCQXNLQztBQUVELE1BQU0sTUFBTSxHQUFHLElBQUksUUFBUSxDQUFDLElBQUksQ0FBQyxDQUFBO0FBQ3hCLHdCQUFNIn0=