import ts from 'typescript';
import { VSCodeEmmetConfig } from '@vscode/emmet-helper';
import { Document } from './lib/documents';
import { ClientCapabilities } from 'vscode-languageserver-protocol';
/**
 * Representation of the language server config.
 * Should be kept in sync with infos in `packages/svelte-vscode/package.json`.
 */
export interface LSConfig {
    typescript: LSTypescriptConfig;
    css: LSCSSConfig;
    html: LSHTMLConfig;
    svelte: LSSvelteConfig;
}
export interface LSTypescriptConfig {
    enable: boolean;
    diagnostics: {
        enable: boolean;
    };
    hover: {
        enable: boolean;
    };
    documentSymbols: {
        enable: boolean;
    };
    completions: {
        enable: boolean;
    };
    codeActions: {
        enable: boolean;
    };
    selectionRange: {
        enable: boolean;
    };
    signatureHelp: {
        enable: boolean;
    };
    semanticTokens: {
        enable: boolean;
    };
}
export interface LSCSSConfig {
    enable: boolean;
    globals: string;
    diagnostics: {
        enable: boolean;
    };
    hover: {
        enable: boolean;
    };
    completions: {
        enable: boolean;
        emmet: boolean;
    };
    documentColors: {
        enable: boolean;
    };
    colorPresentations: {
        enable: boolean;
    };
    documentSymbols: {
        enable: boolean;
    };
    selectionRange: {
        enable: boolean;
    };
}
export interface LSHTMLConfig {
    enable: boolean;
    hover: {
        enable: boolean;
    };
    completions: {
        enable: boolean;
        emmet: boolean;
    };
    tagComplete: {
        enable: boolean;
    };
    documentSymbols: {
        enable: boolean;
    };
    linkedEditing: {
        enable: boolean;
    };
}
export type CompilerWarningsSettings = Record<string, 'ignore' | 'error'>;
export interface LSSvelteConfig {
    enable: boolean;
    compilerWarnings: CompilerWarningsSettings;
    diagnostics: {
        enable: boolean;
    };
    format: {
        enable: boolean;
        config: {
            svelteSortOrder: string;
            svelteStrictMode: boolean;
            svelteAllowShorthand: boolean;
            svelteBracketNewLine: boolean;
            svelteIndentScriptAndStyle: boolean;
            printWidth: number;
            singleQuote: boolean;
        };
    };
    rename: {
        enable: boolean;
    };
    completions: {
        enable: boolean;
    };
    hover: {
        enable: boolean;
    };
    codeActions: {
        enable: boolean;
    };
    selectionRange: {
        enable: boolean;
    };
    defaultScriptLanguage: 'none' | 'ts';
}
/**
 * A subset of the JS/TS VS Code settings which
 * are transformed to ts.UserPreferences.
 * It may not be available in other IDEs, that's why the keys are optional.
 */
export interface TSUserConfig {
    preferences?: TsUserPreferencesConfig;
    suggest?: TSSuggestConfig;
    format?: TsFormatConfig;
    inlayHints?: TsInlayHintsConfig;
    referencesCodeLens?: TsReferenceCodeLensConfig;
    implementationsCodeLens?: TsImplementationCodeLensConfig;
}
/**
 * A subset of the JS/TS VS Code settings which
 * are transformed to ts.UserPreferences.
 */
export interface TsUserPreferencesConfig {
    importModuleSpecifier: ts.UserPreferences['importModuleSpecifierPreference'];
    importModuleSpecifierEnding: ts.UserPreferences['importModuleSpecifierEnding'];
    quoteStyle: ts.UserPreferences['quotePreference'];
    autoImportFileExcludePatterns: ts.UserPreferences['autoImportFileExcludePatterns'];
    /**
     * only in typescript config
     */
    includePackageJsonAutoImports?: ts.UserPreferences['includePackageJsonAutoImports'];
    preferTypeOnlyAutoImports?: ts.UserPreferences['preferTypeOnlyAutoImports'];
}
/**
 * A subset of the JS/TS VS Code settings which
 * are transformed to ts.UserPreferences.
 */
export interface TSSuggestConfig {
    autoImports: ts.UserPreferences['includeCompletionsForModuleExports'];
    includeAutomaticOptionalChainCompletions: boolean | undefined;
    includeCompletionsForImportStatements: boolean | undefined;
    classMemberSnippets: {
        enabled: boolean;
    } | undefined;
    objectLiteralMethodSnippets: {
        enabled: boolean;
    } | undefined;
    includeCompletionsWithSnippetText: boolean | undefined;
}
export type TsFormatConfig = Omit<ts.FormatCodeSettings, 'indentMultiLineObjectLiteralBeginningOnBlankLine' | keyof ts.EditorSettings>;
export interface TsInlayHintsConfig {
    enumMemberValues: {
        enabled: boolean;
    } | undefined;
    functionLikeReturnTypes: {
        enabled: boolean;
    } | undefined;
    parameterNames: {
        enabled: ts.UserPreferences['includeInlayParameterNameHints'];
        suppressWhenArgumentMatchesName: boolean;
    } | undefined;
    parameterTypes: {
        enabled: boolean;
    } | undefined;
    propertyDeclarationTypes: {
        enabled: boolean;
    } | undefined;
    variableTypes: {
        enabled: boolean;
        suppressWhenTypeMatchesName: boolean;
    } | undefined;
}
export interface TsReferenceCodeLensConfig {
    showOnAllFunctions?: boolean | undefined;
    enabled: boolean;
}
export interface TsImplementationCodeLensConfig {
    enabled: boolean;
    showOnInterfaceMethods?: boolean | undefined;
}
export type TsUserConfigLang = 'typescript' | 'javascript';
/**
 * The config as the vscode-css-languageservice understands it
 */
export interface CssConfig {
    validate?: boolean;
    lint?: any;
    completion?: any;
    hover?: any;
}
/**
 * The config as the vscode-html-languageservice understands it
 */
export interface HTMLConfig {
    customData?: string[];
}
type DeepPartial<T> = T extends CompilerWarningsSettings ? T : {
    [P in keyof T]?: DeepPartial<T[P]>;
};
export declare class LSConfigManager {
    private config;
    private listeners;
    private tsUserPreferences;
    private rawTsUserConfig;
    private resolvedAutoImportExcludeCache;
    private tsFormatCodeOptions;
    private prettierConfig;
    private emmetConfig;
    private cssConfig;
    private scssConfig;
    private lessConfig;
    private htmlConfig;
    private isTrusted;
    private clientCapabilities;
    constructor();
    /**
     * Updates config.
     */
    update(config: DeepPartial<LSConfig> | undefined): void;
    /**
     * Whether or not specified config is enabled
     * @param key a string which is a path. Example: 'svelte.diagnostics.enable'.
     */
    enabled(key: string): boolean;
    /**
     * Get specific config
     * @param key a string which is a path. Example: 'svelte.diagnostics.enable'.
     */
    get<T>(key: string): T;
    /**
     * Get the whole config
     */
    getConfig(): Readonly<LSConfig>;
    /**
     * Register a listener which is invoked when the config changed.
     */
    onChange(callback: (config: LSConfigManager) => void): void;
    updateEmmetConfig(config: VSCodeEmmetConfig): void;
    getEmmetConfig(): VSCodeEmmetConfig;
    updatePrettierConfig(config: any): void;
    getPrettierConfig(): any;
    /**
     * Returns a merged Prettier config following these rules:
     * - If `prettierFromFileConfig` exists, that one is returned
     * - Else the Svelte extension's Prettier config is used as a starting point,
     *   and overridden by a possible Prettier config from the Prettier extension,
     *   or, if that doesn't exist, a possible fallback override.
     */
    getMergedPrettierConfig(prettierFromFileConfig: any, overridesWhenNoPrettierConfig?: any): any;
    updateTsJsUserPreferences(config: Record<TsUserConfigLang, TSUserConfig>): void;
    /**
     * Whether or not the current workspace can be trusted.
     * If not, certain operations should be disabled.
     */
    getIsTrusted(): boolean;
    updateIsTrusted(isTrusted: boolean): void;
    private _updateTsUserPreferences;
    getTsUserPreferences(lang: TsUserConfigLang, workspacePath: string | null): ts.UserPreferences;
    getClientTsUserConfig(lang: TsUserConfigLang): TSUserConfig;
    updateCssConfig(config: CssConfig | undefined): void;
    getCssConfig(): CssConfig | undefined;
    updateScssConfig(config: CssConfig | undefined): void;
    getScssConfig(): CssConfig | undefined;
    updateLessConfig(config: CssConfig | undefined): void;
    getLessConfig(): CssConfig | undefined;
    updateHTMLConfig(config: HTMLConfig | undefined): void;
    getHTMLConfig(): HTMLConfig | undefined;
    updateTsJsFormateConfig(config: Record<TsUserConfigLang, TSUserConfig>): void;
    private getDefaultFormatCodeOptions;
    private _updateTsFormatConfig;
    getFormatCodeSettingsForFile(document: Document, scriptKind: ts.ScriptKind): Promise<ts.FormatCodeSettings>;
    private scheduledUpdate;
    private notifyListeners;
    updateClientCapabilities(clientCapabilities: ClientCapabilities): void;
    getClientCapabilities(): ClientCapabilities | undefined;
}
export {};
