import ts from 'typescript';
import { RelativePattern, TextDocumentContentChangeEvent } from 'vscode-languageserver-protocol';
import { Document } from '../../lib/documents';
import { DocumentSnapshot } from './DocumentSnapshot';
import { GlobalSnapshotsManager, SnapshotManager } from './SnapshotManager';
import { ProjectService } from './serviceCache';
export interface LanguageServiceContainer {
    readonly tsconfigPath: string;
    readonly compilerOptions: ts.CompilerOptions;
    readonly configErrors: ts.Diagnostic[];
    /**
     * @internal Public for tests only
     */
    readonly snapshotManager: SnapshotManager;
    getService(skipSynchronize?: boolean): ts.LanguageService;
    updateSnapshot(documentOrFilePath: Document | string): DocumentSnapshot;
    deleteSnapshot(filePath: string): void;
    invalidateModuleCache(filePath: string[]): void;
    scheduleProjectFileUpdate(watcherNewFiles: string[]): void;
    updateTsOrJsFile(fileName: string, changes?: TextDocumentContentChangeEvent[]): void;
    /**
     * Checks if a file is present in the project.
     * Unlike `fileBelongsToProject`, this doesn't run a file search on disk.
     */
    hasFile(filePath: string): boolean;
    /**
     * Careful, don't call often, or it will hurt performance.
     * Only works for TS versions that have ScriptKind.Deferred
     */
    fileBelongsToProject(filePath: string, isNew: boolean): boolean;
    onAutoImportProviderSettingsChanged(): void;
    onPackageJsonChange(packageJsonPath: string): void;
    getTsConfigSvelteOptions(): {
        namespace: string;
    };
    dispose(): void;
}
declare module 'typescript' {
    interface LanguageServiceHost {
        /**
         * @internal
         * This is needed for the languageService program to know that there is a new file
         * that might change the module resolution results
         */
        hasInvalidatedResolutions?: (sourceFile: string) => boolean;
        getModuleResolutionCache?(): ts.ModuleResolutionCache;
    }
    interface ResolvedModuleWithFailedLookupLocations {
        /** @internal */
        failedLookupLocations?: string[];
        /** @internal */
        affectingLocations?: string[];
        /** @internal */
        resolutionDiagnostics?: ts.Diagnostic[];
        /**
         * @internal
         * Used to issue a diagnostic if typings for a non-relative import couldn't be found
         * while respecting package.json `exports`, but were found when disabling `exports`.
         */
        node10Result?: string;
    }
}
/**
 * For testing only: Reset the cache for services.
 * Try to refactor this some day so that this file provides
 * a setup function which creates all this nicely instead.
 */
export declare function __resetCache(): void;
export interface LanguageServiceDocumentContext {
    ambientTypesSource: string;
    transformOnTemplateError: boolean;
    createDocument: (fileName: string, content: string) => Document;
    globalSnapshotsManager: GlobalSnapshotsManager;
    notifyExceedSizeLimit: (() => void) | undefined;
    extendedConfigCache: Map<string, ts.ExtendedConfigCacheEntry>;
    onProjectReloaded: (() => void) | undefined;
    watchTsConfig: boolean;
    tsSystem: ts.System;
    projectService: ProjectService | undefined;
    watchDirectory: ((patterns: RelativePattern[]) => void) | undefined;
    nonRecursiveWatchPattern: string | undefined;
}
export declare function getService(path: string, workspaceUris: string[], docContext: LanguageServiceDocumentContext): Promise<LanguageServiceContainer>;
export declare function forAllServices(cb: (service: LanguageServiceContainer) => any): Promise<void>;
/**
 * @param tsconfigPath has to be absolute
 * @param docContext
 */
export declare function getServiceForTsconfig(tsconfigPath: string, workspacePath: string, docContext: LanguageServiceDocumentContext): Promise<LanguageServiceContainer>;
