"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.create = create;
const language_core_1 = require("@vue/language-core");
const shared_1 = require("@vue/shared");
const common_1 = require("@vue/typescript-plugin/lib/common");
const volar_service_html_1 = require("volar-service-html");
const volar_service_pug_1 = require("volar-service-pug");
const html = require("vscode-html-languageservice");
const vscode_uri_1 = require("vscode-uri");
const nameCasing_1 = require("../ideFeatures/nameCasing");
const types_1 = require("../types");
const data_1 = require("./data");
let builtInData;
let modelData;
function create(mode, ts, getTsPluginClient) {
    let customData = [];
    let extraCustomData = [];
    let lastCompletionComponentNames = new Set();
    const onDidChangeCustomDataListeners = new Set();
    const onDidChangeCustomData = (listener) => {
        onDidChangeCustomDataListeners.add(listener);
        return {
            dispose() {
                onDidChangeCustomDataListeners.delete(listener);
            },
        };
    };
    const baseService = mode === 'pug'
        ? (0, volar_service_pug_1.create)({
            getCustomData() {
                return [
                    ...customData,
                    ...extraCustomData,
                ];
            },
            onDidChangeCustomData,
        })
        : (0, volar_service_html_1.create)({
            documentSelector: ['html', 'markdown'],
            getCustomData() {
                return [
                    ...customData,
                    ...extraCustomData,
                ];
            },
            onDidChangeCustomData,
        });
    return {
        name: `vue-template (${mode})`,
        capabilities: {
            ...baseService.capabilities,
            completionProvider: {
                triggerCharacters: [
                    ...baseService.capabilities.completionProvider?.triggerCharacters ?? [],
                    '@', // vue event shorthand
                ],
            },
            inlayHintProvider: {},
            hoverProvider: true,
            diagnosticProvider: {},
            semanticTokensProvider: {
                legend: {
                    tokenTypes: ['class'],
                    tokenModifiers: [],
                },
            }
        },
        create(context) {
            const tsPluginClient = getTsPluginClient?.(context);
            const baseServiceInstance = baseService.create(context);
            builtInData ??= (0, data_1.loadTemplateData)(context.env.locale ?? 'en');
            modelData ??= (0, data_1.loadModelModifiersData)(context.env.locale ?? 'en');
            // https://vuejs.org/api/built-in-directives.html#v-on
            // https://vuejs.org/api/built-in-directives.html#v-bind
            const eventModifiers = {};
            const propModifiers = {};
            const vOn = builtInData.globalAttributes?.find(x => x.name === 'v-on');
            const vBind = builtInData.globalAttributes?.find(x => x.name === 'v-bind');
            if (vOn) {
                const markdown = (typeof vOn.description === 'string' ? vOn.description : vOn.description?.value) ?? '';
                const modifiers = markdown
                    .split('\n- ')[4]
                    .split('\n').slice(2, -1);
                for (let text of modifiers) {
                    text = text.substring('  - `.'.length);
                    const [name, disc] = text.split('` - ');
                    eventModifiers[name] = disc;
                }
            }
            if (vBind) {
                const markdown = (typeof vBind.description === 'string' ? vBind.description : vBind.description?.value) ?? '';
                const modifiers = markdown
                    .split('\n- ')[4]
                    .split('\n').slice(2, -1);
                for (let text of modifiers) {
                    text = text.substring('  - `.'.length);
                    const [name, disc] = text.split('` - ');
                    propModifiers[name] = disc;
                }
            }
            const disposable = context.env.onDidChangeConfiguration?.(() => initializing = undefined);
            let initializing;
            return {
                ...baseServiceInstance,
                dispose() {
                    baseServiceInstance.dispose?.();
                    disposable?.dispose();
                },
                async provideCompletionItems(document, position, completionContext, token) {
                    if (!isSupportedDocument(document)) {
                        return;
                    }
                    if (!context.project.vue) {
                        return;
                    }
                    const vueCompilerOptions = context.project.vue.compilerOptions;
                    let sync;
                    let currentVersion;
                    const decoded = context.decodeEmbeddedDocumentUri(vscode_uri_1.URI.parse(document.uri));
                    const sourceScript = decoded && context.language.scripts.get(decoded[0]);
                    if (sourceScript?.generated?.root instanceof language_core_1.VueVirtualCode) {
                        // #4298: Precompute HTMLDocument before provideHtmlData to avoid parseHTMLDocument requesting component names from tsserver
                        baseServiceInstance.provideCompletionItems?.(document, position, completionContext, token);
                        sync = (await provideHtmlData(vueCompilerOptions, sourceScript.id, sourceScript.generated.root)).sync;
                        currentVersion = await sync();
                    }
                    let htmlComplete = await baseServiceInstance.provideCompletionItems?.(document, position, completionContext, token);
                    while (currentVersion !== (currentVersion = await sync?.())) {
                        htmlComplete = await baseServiceInstance.provideCompletionItems?.(document, position, completionContext, token);
                    }
                    if (!htmlComplete) {
                        return;
                    }
                    if (sourceScript?.generated?.root instanceof language_core_1.VueVirtualCode) {
                        await afterHtmlCompletion(htmlComplete, context.documents.get(sourceScript.id, sourceScript.languageId, sourceScript.snapshot));
                    }
                    return htmlComplete;
                },
                async provideInlayHints(document) {
                    if (!isSupportedDocument(document)) {
                        return;
                    }
                    if (!context.project.vue) {
                        return;
                    }
                    const vueCompilerOptions = context.project.vue.compilerOptions;
                    const enabled = await context.env.getConfiguration?.('vue.inlayHints.missingProps') ?? false;
                    if (!enabled) {
                        return;
                    }
                    const result = [];
                    const uri = vscode_uri_1.URI.parse(document.uri);
                    const decoded = context.decodeEmbeddedDocumentUri(uri);
                    const sourceScript = decoded && context.language.scripts.get(decoded[0]);
                    const virtualCode = decoded && sourceScript?.generated?.embeddedCodes.get(decoded[1]);
                    if (!virtualCode) {
                        return;
                    }
                    const code = context.language.scripts.get(decoded[0])?.generated?.root;
                    const scanner = getScanner(baseServiceInstance, document);
                    if (code instanceof language_core_1.VueVirtualCode && scanner) {
                        // visualize missing required props
                        const casing = await (0, nameCasing_1.getNameCasing)(context, decoded[0]);
                        const components = await tsPluginClient?.getComponentNames(code.fileName) ?? [];
                        const componentProps = {};
                        let token;
                        let current;
                        while ((token = scanner.scan()) !== html.TokenType.EOS) {
                            if (token === html.TokenType.StartTag) {
                                const tagName = scanner.getTokenText();
                                const checkTag = tagName.indexOf('.') >= 0
                                    ? tagName
                                    : components.find(component => component === tagName || (0, language_core_1.hyphenateTag)(component) === tagName);
                                if (checkTag) {
                                    componentProps[checkTag] ??= await tsPluginClient?.getComponentProps(code.fileName, checkTag, true) ?? [];
                                    current = {
                                        unburnedRequiredProps: [...componentProps[checkTag]],
                                        labelOffset: scanner.getTokenOffset() + scanner.getTokenLength(),
                                        insertOffset: scanner.getTokenOffset() + scanner.getTokenLength(),
                                    };
                                }
                            }
                            else if (token === html.TokenType.AttributeName) {
                                if (current) {
                                    let attrText = scanner.getTokenText();
                                    if (attrText === 'v-bind') {
                                        current.unburnedRequiredProps = [];
                                    }
                                    else {
                                        // remove modifiers
                                        if (attrText.indexOf('.') >= 0) {
                                            attrText = attrText.split('.')[0];
                                        }
                                        // normalize
                                        if (attrText.startsWith('v-bind:')) {
                                            attrText = attrText.substring('v-bind:'.length);
                                        }
                                        else if (attrText.startsWith(':')) {
                                            attrText = attrText.substring(':'.length);
                                        }
                                        else if (attrText.startsWith('v-model:')) {
                                            attrText = attrText.substring('v-model:'.length);
                                        }
                                        else if (attrText === 'v-model') {
                                            attrText = vueCompilerOptions.target >= 3 ? 'modelValue' : 'value'; // TODO: support for experimentalModelPropName?
                                        }
                                        else if (attrText.startsWith('@')) {
                                            attrText = 'on-' + (0, language_core_1.hyphenateAttr)(attrText.substring('@'.length));
                                        }
                                        current.unburnedRequiredProps = current.unburnedRequiredProps.filter(propName => {
                                            return attrText !== propName
                                                && attrText !== (0, language_core_1.hyphenateAttr)(propName);
                                        });
                                    }
                                }
                            }
                            else if (token === html.TokenType.StartTagSelfClose || token === html.TokenType.StartTagClose) {
                                if (current) {
                                    for (const requiredProp of current.unburnedRequiredProps) {
                                        result.push({
                                            label: `${requiredProp}!`,
                                            paddingLeft: true,
                                            position: document.positionAt(current.labelOffset),
                                            kind: 2,
                                            textEdits: [{
                                                    range: {
                                                        start: document.positionAt(current.insertOffset),
                                                        end: document.positionAt(current.insertOffset),
                                                    },
                                                    newText: ` :${casing.attr === types_1.AttrNameCasing.Kebab ? (0, language_core_1.hyphenateAttr)(requiredProp) : requiredProp}=`,
                                                }],
                                        });
                                    }
                                    current = undefined;
                                }
                            }
                            if (token === html.TokenType.AttributeName || token === html.TokenType.AttributeValue) {
                                if (current) {
                                    current.insertOffset = scanner.getTokenOffset() + scanner.getTokenLength();
                                }
                            }
                        }
                    }
                    return result;
                },
                provideHover(document, position, token) {
                    if (!isSupportedDocument(document)) {
                        return;
                    }
                    if (context.decodeEmbeddedDocumentUri(vscode_uri_1.URI.parse(document.uri))) {
                        updateExtraCustomData([]);
                    }
                    return baseServiceInstance.provideHover?.(document, position, token);
                },
                async provideDiagnostics(document, token) {
                    if (!isSupportedDocument(document)) {
                        return;
                    }
                    const originalResult = await baseServiceInstance.provideDiagnostics?.(document, token);
                    const uri = vscode_uri_1.URI.parse(document.uri);
                    const decoded = context.decodeEmbeddedDocumentUri(uri);
                    const sourceScript = decoded && context.language.scripts.get(decoded[0]);
                    const virtualCode = decoded && sourceScript?.generated?.embeddedCodes.get(decoded[1]);
                    if (!virtualCode) {
                        return;
                    }
                    const code = context.language.scripts.get(decoded[0])?.generated?.root;
                    if (!(code instanceof language_core_1.VueVirtualCode)) {
                        return;
                    }
                    const templateErrors = [];
                    const { template } = code.sfc;
                    if (template) {
                        for (const error of template.errors) {
                            onCompilerError(error, 1);
                        }
                        for (const warning of template.warnings) {
                            onCompilerError(warning, 2);
                        }
                        function onCompilerError(error, severity) {
                            const templateHtmlRange = {
                                start: error.loc?.start.offset ?? 0,
                                end: error.loc?.end.offset ?? 0,
                            };
                            let errorMessage = error.message;
                            templateErrors.push({
                                range: {
                                    start: document.positionAt(templateHtmlRange.start),
                                    end: document.positionAt(templateHtmlRange.end),
                                },
                                severity,
                                code: error.code,
                                source: 'vue',
                                message: errorMessage,
                            });
                        }
                    }
                    return [
                        ...originalResult ?? [],
                        ...templateErrors,
                    ];
                },
                provideDocumentSemanticTokens(document, range, legend) {
                    if (!isSupportedDocument(document)) {
                        return;
                    }
                    if (!context.project.vue) {
                        return;
                    }
                    const vueCompilerOptions = context.project.vue.compilerOptions;
                    const languageService = context.inject('typescript/languageService');
                    if (!languageService) {
                        return;
                    }
                    const decoded = context.decodeEmbeddedDocumentUri(vscode_uri_1.URI.parse(document.uri));
                    const sourceScript = decoded && context.language.scripts.get(decoded[0]);
                    if (!sourceScript
                        || !(sourceScript.generated?.root instanceof language_core_1.VueVirtualCode)
                        || !sourceScript.generated.root.sfc.template) {
                        return [];
                    }
                    const { template } = sourceScript.generated.root.sfc;
                    const spans = common_1.getComponentSpans.call({
                        files: context.language.scripts,
                        languageService,
                        typescript: ts,
                        vueOptions: vueCompilerOptions,
                    }, sourceScript.generated.root, template, {
                        start: document.offsetAt(range.start),
                        length: document.offsetAt(range.end) - document.offsetAt(range.start),
                    });
                    const classTokenIndex = legend.tokenTypes.indexOf('class');
                    return spans.map(span => {
                        const start = document.positionAt(span.start);
                        return [
                            start.line,
                            start.character,
                            span.length,
                            classTokenIndex,
                            0,
                        ];
                    });
                },
            };
            async function provideHtmlData(vueCompilerOptions, sourceDocumentUri, vueCode) {
                await (initializing ??= initialize());
                const casing = await (0, nameCasing_1.getNameCasing)(context, sourceDocumentUri);
                if (builtInData.tags) {
                    for (const tag of builtInData.tags) {
                        if (tag.name === 'slot') {
                            continue;
                        }
                        if (tag.name === 'component') {
                            continue;
                        }
                        if (tag.name === 'template') {
                            continue;
                        }
                        if (casing.tag === types_1.TagNameCasing.Kebab) {
                            tag.name = (0, language_core_1.hyphenateTag)(tag.name);
                        }
                        else {
                            tag.name = (0, shared_1.camelize)((0, shared_1.capitalize)(tag.name));
                        }
                    }
                }
                const promises = [];
                const tagInfos = new Map();
                let version = 0;
                let components;
                let templateContextProps;
                updateExtraCustomData([
                    html.newHTMLDataProvider('vue-template-built-in', builtInData),
                    {
                        getId: () => 'vue-template',
                        isApplicable: () => true,
                        provideTags: () => {
                            if (!components) {
                                promises.push((async () => {
                                    components = (await tsPluginClient?.getComponentNames(vueCode.fileName) ?? [])
                                        .filter(name => name !== 'Transition'
                                        && name !== 'TransitionGroup'
                                        && name !== 'KeepAlive'
                                        && name !== 'Suspense'
                                        && name !== 'Teleport');
                                    lastCompletionComponentNames = new Set(components);
                                    version++;
                                })());
                                return [];
                            }
                            const scriptSetupRanges = vueCode.sfc.scriptSetup
                                ? (0, language_core_1.parseScriptSetupRanges)(ts, vueCode.sfc.scriptSetup.ast, vueCompilerOptions)
                                : undefined;
                            const names = new Set();
                            const tags = [];
                            for (const tag of components) {
                                if (casing.tag === types_1.TagNameCasing.Kebab) {
                                    names.add((0, language_core_1.hyphenateTag)(tag));
                                }
                                else if (casing.tag === types_1.TagNameCasing.Pascal) {
                                    names.add(tag);
                                }
                            }
                            for (const binding of scriptSetupRanges?.bindings ?? []) {
                                const name = vueCode.sfc.scriptSetup.content.substring(binding.start, binding.end);
                                if (casing.tag === types_1.TagNameCasing.Kebab) {
                                    names.add((0, language_core_1.hyphenateTag)(name));
                                }
                                else if (casing.tag === types_1.TagNameCasing.Pascal) {
                                    names.add(name);
                                }
                            }
                            for (const name of names) {
                                tags.push({
                                    name: name,
                                    attributes: [],
                                });
                            }
                            return tags;
                        },
                        provideAttributes: tag => {
                            const tagInfo = tagInfos.get(tag);
                            if (!tagInfo) {
                                promises.push((async () => {
                                    const attrs = await tsPluginClient?.getElementAttrs(vueCode.fileName, tag) ?? [];
                                    const props = await tsPluginClient?.getComponentProps(vueCode.fileName, tag) ?? [];
                                    const events = await tsPluginClient?.getComponentEvents(vueCode.fileName, tag) ?? [];
                                    tagInfos.set(tag, {
                                        attrs,
                                        props: props.filter(prop => !prop.startsWith('ref_')
                                            && !(0, shared_1.hyphenate)(prop).startsWith('on-vnode-')),
                                        events,
                                    });
                                    version++;
                                })());
                                return [];
                            }
                            const { attrs, props, events } = tagInfo;
                            const attributes = [];
                            const _tsCodegen = language_core_1.tsCodegen.get(vueCode.sfc);
                            if (_tsCodegen) {
                                if (!templateContextProps) {
                                    promises.push((async () => {
                                        templateContextProps = await tsPluginClient?.getTemplateContextProps(vueCode.fileName) ?? [];
                                        version++;
                                    })());
                                    return [];
                                }
                                let ctxVars = [
                                    ..._tsCodegen.scriptRanges()?.bindings.map(binding => vueCode.sfc.script.content.substring(binding.start, binding.end)) ?? [],
                                    ..._tsCodegen.scriptSetupRanges()?.bindings.map(binding => vueCode.sfc.scriptSetup.content.substring(binding.start, binding.end)) ?? [],
                                    ...templateContextProps,
                                ];
                                ctxVars = [...new Set(ctxVars)];
                                const dirs = ctxVars.map(language_core_1.hyphenateAttr).filter(v => v.startsWith('v-'));
                                for (const dir of dirs) {
                                    attributes.push({
                                        name: dir,
                                    });
                                }
                            }
                            const propsSet = new Set(props);
                            for (const prop of [...props, ...attrs]) {
                                const isGlobal = !propsSet.has(prop);
                                const name = casing.attr === types_1.AttrNameCasing.Camel ? prop : (0, language_core_1.hyphenateAttr)(prop);
                                if ((0, language_core_1.hyphenateAttr)(name).startsWith('on-')) {
                                    const propNameBase = name.startsWith('on-')
                                        ? name.slice('on-'.length)
                                        : (name['on'.length].toLowerCase() + name.slice('onX'.length));
                                    const propKey = createInternalItemId('componentEvent', [isGlobal ? '*' : tag, propNameBase]);
                                    attributes.push({
                                        name: 'v-on:' + propNameBase,
                                        description: propKey,
                                    }, {
                                        name: '@' + propNameBase,
                                        description: propKey,
                                    });
                                }
                                {
                                    const propName = name;
                                    const propKey = createInternalItemId('componentProp', [isGlobal ? '*' : tag, propName]);
                                    attributes.push({
                                        name: propName,
                                        description: propKey,
                                    }, {
                                        name: ':' + propName,
                                        description: propKey,
                                    }, {
                                        name: 'v-bind:' + propName,
                                        description: propKey,
                                    });
                                }
                            }
                            for (const event of events) {
                                const name = casing.attr === types_1.AttrNameCasing.Camel ? event : (0, language_core_1.hyphenateAttr)(event);
                                const propKey = createInternalItemId('componentEvent', [tag, name]);
                                attributes.push({
                                    name: 'v-on:' + name,
                                    description: propKey,
                                });
                                attributes.push({
                                    name: '@' + name,
                                    description: propKey,
                                });
                            }
                            const models = [];
                            for (const prop of [...props, ...attrs]) {
                                if (prop.startsWith('onUpdate:')) {
                                    const isGlobal = !propsSet.has(prop);
                                    models.push([isGlobal, prop.substring('onUpdate:'.length)]);
                                }
                            }
                            for (const event of events) {
                                if (event.startsWith('update:')) {
                                    models.push([false, event.substring('update:'.length)]);
                                }
                            }
                            for (const [isGlobal, model] of models) {
                                const name = casing.attr === types_1.AttrNameCasing.Camel ? model : (0, language_core_1.hyphenateAttr)(model);
                                const propKey = createInternalItemId('componentProp', [isGlobal ? '*' : tag, name]);
                                attributes.push({
                                    name: 'v-model:' + name,
                                    description: propKey,
                                });
                                if (model === 'modelValue') {
                                    attributes.push({
                                        name: 'v-model',
                                        description: propKey,
                                    });
                                }
                            }
                            return attributes;
                        },
                        provideValues: () => [],
                    },
                ]);
                return {
                    async sync() {
                        await Promise.all(promises);
                        return version;
                    }
                };
            }
            function afterHtmlCompletion(completionList, sourceDocument) {
                const replacement = getReplacement(completionList, sourceDocument);
                if (replacement) {
                    const isEvent = replacement.text.startsWith('v-on:') || replacement.text.startsWith('@');
                    const isProp = replacement.text.startsWith('v-bind:') || replacement.text.startsWith(':');
                    const isModel = replacement.text.startsWith('v-model:') || replacement.text.split('.')[0] === 'v-model';
                    const hasModifier = replacement.text.includes('.');
                    const validModifiers = isEvent ? eventModifiers
                        : isProp ? propModifiers
                            : undefined;
                    const modifiers = replacement.text.split('.').slice(1);
                    const textWithoutModifier = replacement.text.split('.')[0];
                    if (validModifiers && hasModifier) {
                        for (const modifier in validModifiers) {
                            if (modifiers.includes(modifier)) {
                                continue;
                            }
                            const modifierDes = validModifiers[modifier];
                            const insertText = textWithoutModifier + modifiers.slice(0, -1).map(m => '.' + m).join('') + '.' + modifier;
                            const newItem = {
                                label: modifier,
                                filterText: insertText,
                                documentation: {
                                    kind: 'markdown',
                                    value: modifierDes,
                                },
                                textEdit: {
                                    range: replacement.textEdit.range,
                                    newText: insertText,
                                },
                                kind: 20,
                            };
                            completionList.items.push(newItem);
                        }
                    }
                    else if (hasModifier && isModel) {
                        for (const modifier of modelData.globalAttributes ?? []) {
                            if (modifiers.includes(modifier.name)) {
                                continue;
                            }
                            const insertText = textWithoutModifier + modifiers.slice(0, -1).map(m => '.' + m).join('') + '.' + modifier.name;
                            const newItem = {
                                label: modifier.name,
                                filterText: insertText,
                                documentation: {
                                    kind: 'markdown',
                                    value: (typeof modifier.description === 'object' ? modifier.description.value : modifier.description)
                                        + '\n\n' + modifier.references?.map(ref => `[${ref.name}](${ref.url})`).join(' | '),
                                },
                                textEdit: {
                                    range: replacement.textEdit.range,
                                    newText: insertText,
                                },
                                kind: 20,
                            };
                            completionList.items.push(newItem);
                        }
                    }
                }
                for (const item of completionList.items) {
                    const itemIdKey = typeof item.documentation === 'string' ? item.documentation : item.documentation?.value;
                    const itemId = itemIdKey ? readInternalItemId(itemIdKey) : undefined;
                    if (itemId) {
                        item.documentation = undefined;
                    }
                    if (item.kind === 10 && lastCompletionComponentNames.has((0, language_core_1.hyphenateTag)(item.label))) {
                        item.kind = 6;
                        item.sortText = '\u0000' + (item.sortText ?? item.label);
                    }
                    else if (itemId && (itemId.type === 'componentProp' || itemId.type === 'componentEvent')) {
                        const [componentName] = itemId.args;
                        if (componentName !== '*') {
                            if (item.label === 'class'
                                || item.label === 'ref'
                                || item.label.endsWith(':class')
                                || item.label.endsWith(':ref')) {
                                item.sortText = '\u0000' + (item.sortText ?? item.label);
                            }
                            else {
                                item.sortText = '\u0000\u0000' + (item.sortText ?? item.label);
                            }
                        }
                        if (itemId.type === 'componentProp') {
                            if (componentName !== '*') {
                                item.kind = 5;
                            }
                        }
                        else {
                            item.kind = componentName !== '*' ? 3 : 23;
                        }
                    }
                    else if (item.label === 'v-if'
                        || item.label === 'v-else-if'
                        || item.label === 'v-else'
                        || item.label === 'v-for') {
                        item.kind = 14;
                        item.sortText = '\u0003' + (item.sortText ?? item.label);
                    }
                    else if (item.label.startsWith('v-')) {
                        item.kind = 3;
                        item.sortText = '\u0002' + (item.sortText ?? item.label);
                    }
                    else {
                        item.sortText = '\u0001' + (item.sortText ?? item.label);
                    }
                }
                updateExtraCustomData([]);
            }
            async function initialize() {
                customData = await getHtmlCustomData();
            }
            async function getHtmlCustomData() {
                const customData = await context.env.getConfiguration?.('html.customData') ?? [];
                const newData = [];
                for (const customDataPath of customData) {
                    for (const workspaceFolder of context.env.workspaceFolders) {
                        const uri = vscode_uri_1.Utils.resolvePath(workspaceFolder, customDataPath);
                        const json = await context.env.fs?.readFile?.(uri);
                        if (json) {
                            try {
                                const data = JSON.parse(json);
                                newData.push(html.newHTMLDataProvider(customDataPath, data));
                            }
                            catch (error) {
                                console.error(error);
                            }
                        }
                    }
                }
                return newData;
            }
        },
    };
    function getScanner(service, document) {
        if (mode === 'html') {
            return service.provide['html/languageService']().createScanner(document.getText());
        }
        else {
            const pugDocument = service.provide['pug/pugDocument'](document);
            if (pugDocument) {
                return service.provide['pug/languageService']().createScanner(pugDocument);
            }
        }
    }
    function updateExtraCustomData(extraData) {
        extraCustomData = extraData;
        onDidChangeCustomDataListeners.forEach(l => l());
    }
    function isSupportedDocument(document) {
        if (mode === 'pug') {
            return document.languageId === 'jade';
        }
        else {
            return document.languageId === 'html';
        }
    }
}
;
function createInternalItemId(type, args) {
    return '__VLS_::' + type + '::' + args.join(',');
}
function readInternalItemId(key) {
    if (key.startsWith('__VLS_::')) {
        const strs = key.split('::');
        return {
            type: strs[1],
            args: strs[2].split(','),
        };
    }
}
function getReplacement(list, doc) {
    for (const item of list.items) {
        if (item.textEdit && 'range' in item.textEdit) {
            return {
                item: item,
                textEdit: item.textEdit,
                text: doc.getText(item.textEdit.range)
            };
        }
    }
}
//# sourceMappingURL=vue-template.js.map