'use strict';

const node_module = require('node:module');

const NodeBuiltinModules = [].concat(
  node_module.Module.builtinModules,
  [
    "assert/strict",
    "fs/promises",
    "path/posix",
    "path/win32",
    "stream/consumers",
    "stream/promises",
    "stream/web",
    "timers/promises",
    "util/types"
  ]
);
function mapArrToVal(val, arr) {
  return Object.fromEntries(arr.map((c) => [c, val]));
}

const node = {
  alias: {
    "node-fetch": "unenv/runtime/npm/node-fetch",
    "cross-fetch": "unenv/runtime/npm/cross-fetch",
    "cross-fetch/polyfill": "unenv/runtime/mock/empty",
    "isomorphic-fetch": "unenv/runtime/mock/empty"
  },
  polyfill: ["node-fetch-native/polyfill"],
  external: [...NodeBuiltinModules]
};

const nodeless = {
  alias: {
    // Generic mock for built-ins
    ...mapArrToVal("unenv/runtime/mock/proxy-cjs", NodeBuiltinModules),
    // Built-ins implemented by unenv
    "buffer/index.js": "buffer",
    ...Object.fromEntries(
      [
        "async_hooks",
        "buffer",
        "console",
        "child_process",
        "constants",
        "cluster",
        "crypto",
        "dgram",
        "diagnostics_channel",
        "dns",
        "dns/promises",
        "domain",
        "events",
        "fs",
        "fs/promises",
        "http",
        "https",
        "http2",
        "inspector",
        "module",
        "net",
        "os",
        "path",
        "punycode",
        "perf_hooks",
        "process",
        "querystring",
        "readline",
        "readline/promises",
        "stream",
        "stream/promises",
        "stream/consumers",
        "stream/web",
        "string_decoder",
        "trace_events",
        "timers",
        "timers/promises",
        "tls",
        "tty",
        "url",
        "util",
        "util/types",
        "v8",
        "vm",
        "wasi",
        "worker_threads",
        "zlib"
      ].map((m) => [m, `unenv/runtime/node/${m}/index`])
    ),
    "path/posix": "unenv/runtime/node/path/index",
    "path/win32": "unenv/runtime/node/path/index",
    "inspector/promises": "unenv/runtime/node/inspector/index",
    // The sys module is deprecated and has been renamed util
    // https://github.com/nodejs/node/blob/main/lib/sys.js#L27
    sys: "unenv/runtime/node/util/index",
    // npm
    etag: "unenv/runtime/mock/noop",
    "mime-db": "unenv/runtime/npm/mime-db",
    mime: "unenv/runtime/npm/mime",
    "mime/lite": "unenv/runtime/npm/mime",
    _mime: "mime/lite.js",
    fsevents: "unenv/runtime/npm/fsevents",
    "consola/core": "consola/core",
    "node-fetch": "unenv/runtime/npm/node-fetch",
    "node-fetch-native": "unenv/runtime/npm/node-fetch",
    "node-fetch-native/polyfill": "unenv/runtime/mock/empty",
    "cross-fetch": "unenv/runtime/npm/cross-fetch",
    "cross-fetch/polyfill": "unenv/runtime/mock/empty",
    "isomorphic-fetch": "unenv/runtime/mock/empty",
    inherits: "unenv/runtime/npm/inherits"
  },
  inject: {
    global: "unenv/runtime/node/_global",
    process: "unenv/runtime/node/process",
    Buffer: ["buffer", "Buffer"],
    performance: "unenv/runtime/polyfill/performance"
  },
  polyfill: [
    "unenv/runtime/polyfill/node-global",
    "unenv/runtime/polyfill/process",
    "unenv/runtime/polyfill/performance"
  ]
};
for (const m of NodeBuiltinModules) {
  nodeless.alias[`node:${m}`] = nodeless.alias[m];
}
const nodeless$1 = nodeless;

const denoNodeCompatModules = [
  "assert",
  "assert/strict",
  "async_hooks",
  "buffer",
  "child_process",
  "cluster",
  "console",
  "constants",
  "crypto",
  "dgram",
  "diagnostics_channel",
  "dns",
  "dns/promises",
  "domain",
  "events",
  "fs",
  "fs/promises",
  "http",
  "http2",
  "https",
  "module",
  "net",
  "os",
  "path",
  "path/posix",
  "path/win32",
  "perf_hooks",
  "process",
  "punycode",
  "querystring",
  "readline",
  "stream",
  "stream/consumers",
  "stream/promises",
  "stream/web",
  "string_decoder",
  "sys",
  "timers",
  "timers/promises",
  "tls",
  "tty",
  "url",
  "util",
  "util/types",
  "v8",
  "vm",
  "worker_threads",
  "zlib"
];
const denoPreset = {
  alias: {
    ...Object.fromEntries(denoNodeCompatModules.map((p) => [p, `node:${p}`])),
    ...Object.fromEntries(
      denoNodeCompatModules.map((p) => [`node:${p}`, `node:${p}`])
    )
  },
  // Deno's listed globals manually tested against deno@1.38.5
  // TODO: missing BroadcastChannel, PerformanceObserverEntryList, PerformanceResourceTiming
  // TODO: global and process
  inject: {
    setImmediate: "node:timers",
    clearImmediate: "node:timers",
    Buffer: "node:buffer",
    PerformanceObserver: "node:perf_hooks"
  },
  polyfill: ["unenv/runtime/polyfill/deno-env"],
  external: denoNodeCompatModules.map((p) => `node:${p}`)
};
const denoPreset$1 = denoPreset;

const cloudflareNodeCompatModules = [
  "_stream_duplex",
  "_stream_passthrough",
  "_stream_readable",
  "_stream_transform",
  "_stream_writable",
  "assert",
  "diagnostics_channel",
  "events",
  "path",
  "process",
  "stream",
  "stream/consumers",
  "stream/promises",
  "stream/web",
  "string_decoder"
];
const hybridNodeCompatModules = [
  "async_hooks",
  "buffer",
  "crypto",
  "process",
  "util",
  "util/types"
];
const cloudflarePreset = {
  alias: {
    ...Object.fromEntries(cloudflareNodeCompatModules.map((p) => [p, p])),
    ...Object.fromEntries(
      cloudflareNodeCompatModules.map((p) => [`node:${p}`, `node:${p}`])
    ),
    // The `node:assert` implementation of workerd uses strict semantics by default
    "assert/strict": "node:assert",
    "node:assert/strict": "node:assert",
    // The `node:sys` module is just a deprecated alias for `node:util` which we implemented using a hybrid polyfill
    sys: "unenv/runtime/node/util/$cloudflare",
    "node:sys": "unenv/runtime/node/util/$cloudflare",
    // define aliases for hybrid modules
    ...Object.fromEntries(
      hybridNodeCompatModules.map((m) => [
        m,
        `unenv/runtime/node/${m}/$cloudflare`
      ])
    ),
    ...Object.fromEntries(
      hybridNodeCompatModules.map((m) => [
        `node:${m}`,
        `unenv/runtime/node/${m}/$cloudflare`
      ])
    )
  },
  inject: {
    // workerd already defines `global` and `Buffer`
    // override the previous presets so that we use the native implementation
    global: false,
    Buffer: false,
    process: "unenv/runtime/node/process/$cloudflare"
  },
  polyfill: [],
  external: [
    ...cloudflareNodeCompatModules.map((p) => `${p}`),
    ...cloudflareNodeCompatModules.map((p) => `node:${p}`)
  ]
};
const cloudflarePreset$1 = cloudflarePreset;

const vercelNodeCompatModules = [
  "async_hooks",
  "events",
  "buffer",
  "assert",
  "util"
];
const vercelPreset = {
  alias: {
    ...Object.fromEntries(vercelNodeCompatModules.map((p) => [p, `node:${p}`])),
    ...Object.fromEntries(
      vercelNodeCompatModules.map((p) => [`node:${p}`, `node:${p}`])
    )
  },
  inject: {},
  polyfill: [],
  external: vercelNodeCompatModules.map((p) => `node:${p}`)
};
const vercelPreset$1 = vercelPreset;

function env(...presets) {
  const _env = {
    alias: {},
    inject: {},
    polyfill: [],
    external: []
  };
  for (const preset of presets) {
    if (preset.alias) {
      const aliases = Object.keys(preset.alias).sort(
        (a, b) => b.split("/").length - a.split("/").length || b.length - a.length
      );
      for (const from of aliases) {
        _env.alias[from] = preset.alias[from];
      }
    }
    if (preset.inject) {
      for (const global in preset.inject) {
        const globalValue = preset.inject[global];
        if (Array.isArray(globalValue)) {
          const [id, ...path] = globalValue;
          _env.inject[global] = [id, ...path];
        } else if (globalValue === false) {
          delete _env.inject[global];
        } else {
          _env.inject[global] = globalValue;
        }
      }
    }
    if (preset.polyfill) {
      _env.polyfill.push(...preset.polyfill.filter(Boolean));
    }
    if (preset.external) {
      _env.external.push(...preset.external);
    }
  }
  return _env;
}

exports.NodeBuiltinModules = NodeBuiltinModules;
exports.cloudflare = cloudflarePreset$1;
exports.deno = denoPreset$1;
exports.env = env;
exports.mapArrToVal = mapArrToVal;
exports.node = node;
exports.nodeless = nodeless$1;
exports.vercel = vercelPreset$1;
